import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import tripsListReducer from 'src/redux/reducers/trips/tripsList/reducer';

import {EProjectName} from 'constants/common';
import {EAppActions} from 'constants/platforms/TPlatforms';

import {EFooterProject} from 'components/Footer/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {fetchTripsList} from 'reducers/trips/tripsList/thunk';
import {setIsNewTrip} from 'reducers/trips/tripPage/actions';

import tripsPageSelector from 'projects/account/pages/TripsPage/selectors/tripsPageSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useIsAuth} from 'utilities/hooks/useIsAuth';
import getPastTripsInitialCount from 'projects/account/pages/TripsPage/utilities/getPastTripsInitialCount';
import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import withReducers from 'containers/withReducers/withReducers';

import LoadingErrorModal from 'projects/trips/components/LoadingErrorModal/LoadingErrorModal';
import TripsPageContent from 'projects/account/pages/TripsPage/components/TripsPageContent/TripsPageContent';
import LayoutDefault, {
    ELayoutBgColor,
} from 'components/Layouts/LayoutDefault/LayoutDefault';
import AccountMenu from 'projects/account/components/AccountMenu/AccountMenu';
import DesktopSupportPhone from 'projects/trips/components/DesktopSupportPhone/DesktopSupportPhone';
import HangingYandexMessenger from 'components/YandexMessenger/components/HangingYandexMessenger/HangingYandexMessenger';

import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import cx from './TripsPage.scss';

const ROOT_QA = 'tripsPage';

interface ITripsPageProps {}

const TripsPage: React.FC<ITripsPageProps> = () => {
    const deviceType = useDeviceType();
    const {isWeb} = usePlatform();
    const coordinator = useCoordinator();

    const dispatch = useDispatch();
    const {
        tripsList,
        tripsListIsLoading,
        tripsListIsFetched,
        tripsListIsError,
    } = useSelector(tripsPageSelector);
    const isAuth = useIsAuth();

    const fetchTripList = useCallback(() => {
        if (!isAuth) {
            return;
        }

        dispatch(
            fetchTripsList({
                pastLimit: getPastTripsInitialCount(deviceType),
            }),
        );
    }, [dispatch, isAuth, deviceType]);

    useEffect(() => {
        if (!tripsListIsFetched) {
            fetchTripList();
        }

        dispatch(setIsNewTrip(false));

        reachGoal(ETripsGoal.TRIPS_PAGE_SHOW);
        coordinator.doAction(EAppActions.REPORT_METRICS_EVENT, {
            name: ETripsGoal.TRIPS_PAGE_SHOW,
        });
    }, []);

    const supportPhone = useMemo(() => <DesktopSupportPhone />, []);

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            project={EProjectName.ACCOUNT}
            bgColor={
                deviceType.isMobile ? ELayoutBgColor.DARK : ELayoutBgColor.LIGHT
            }
            footerType={EFooterProject.TRIPS}
            showNavigation={deviceType.isDesktop}
            showHeader={isWeb}
            showFooter={isWeb}
            showTabBar
            headerBorderBottomType={EHeaderBorderBottomType.NONE}
            footerIncludeSupportBlock
            footerClassName={cx('footer')}
            {...prepareQaAttributes(ROOT_QA)}
        >
            {deviceType.isDesktop && isAuth && (
                <AccountMenu
                    className={cx('accountMenu')}
                    supportPhone={supportPhone}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'accountMenu',
                    })}
                />
            )}

            <TripsPageContent
                activeTrips={tripsList.active.trips}
                pastTrips={tripsList.past.trips}
                isAuth={isAuth}
                isLoading={tripsListIsLoading}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'content',
                })}
            />

            <LoadingErrorModal
                isVisible={tripsListIsError}
                onRefreshClick={fetchTripList}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'loadingErrorModal',
                })}
            />

            <HangingYandexMessenger
                entrypoint="tripsPage"
                metrikaGoal={ETripsGoal.TRIPS_PAGE_CHAT_CLICK}
            />
        </LayoutDefault>
    );
};

export default withReducers([[ELoadableReducer.TRIPS_LIST, tripsListReducer]])(
    TripsPage,
);
