import {FC, memo, ReactNode} from 'react';
import first from 'lodash/first';

import TTripListItem from 'types/trips/TTripListItem/TTripListItem';

import {getTripListItemKey} from 'projects/trips/utilities/getTripListItemKey';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import {IActiveTripsProps} from 'projects/account/pages/TripsPage/components/ActiveTrips/ActiveTrips';
import ActiveTripItem from 'projects/account/pages/TripsPage/components/ActiveTrips/components/DesktopActiveTrips/components/ActiveTripItem/ActiveTripItem';
import EmptyTripsPlaceholder from 'projects/account/pages/TripsPage/components/EmptyTripsPlaceholder/EmptyTripsPlaceholder';
import SearchOrderLink from 'projects/account/pages/TripsPage/components/SearchOrderLink/SearchOrderLink';
import SearchToursOrderLink from 'projects/account/pages/TripsPage/components/SearchToursOrderLink/SearchToursOrderLink';

import cx from './DesktopActiveTrips.scss';

interface IDesktopActiveTripsProps extends IActiveTripsProps {}

const DesktopActiveTrips: FC<IDesktopActiveTripsProps> = props => {
    const {trips} = props;

    const topRowTrip = first(trips);
    const restTrips = trips.slice(1);

    const topRowTripsCount = trips.length > 0 ? 1 : 0;

    const renderTrip = (trip: TTripListItem, size: 'm' | 'l'): ReactNode => {
        const key = getTripListItemKey(trip);

        return (
            <ActiveTripItem
                key={key}
                className={cx('trip')}
                trip={trip}
                size={size}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'activeTrip',
                    key,
                })}
            />
        );
    };

    return (
        <div className={cx('root')}>
            <div className={cx('topRow', `topRow_layout${topRowTripsCount}`)}>
                <div className={cx('topRowTrips')}>
                    {topRowTrip ? (
                        renderTrip(topRowTrip, 'l')
                    ) : (
                        <EmptyTripsPlaceholder
                            className={cx('topRowTrips')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'emptyTripsPlaceholder',
                            })}
                        />
                    )}
                </div>

                <SearchOrderLink
                    className={cx('searchOrderLink')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'searchOrderLink',
                    })}
                />

                <SearchToursOrderLink
                    className={cx('searchToursLink')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'searchToursLink',
                    })}
                />
            </div>

            <div className={cx('restTrips')}>
                {restTrips.map(trip => renderTrip(trip, 'm'))}
            </div>
        </div>
    );
};

export default memo(DesktopActiveTrips);
