import {FC, memo} from 'react';

import TTripListItem from 'types/trips/TTripListItem/TTripListItem';
import {IWithClassName} from 'types/withClassName';
import ETripImageType from 'projects/trips/components/TripImage/types/ETripImageType';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTripOrOrderUrl} from 'projects/account/utilities/urls/getTripOrOrderUrl';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import Flex from 'components/Flex/Flex';
import TripImage from 'projects/trips/components/TripImage/TripImage';
import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import TripBadges from 'projects/account/pages/TripsPage/components/TripBadges/TripBadges';

import cx from './ActiveTripItem.scss';

interface IActiveTripItemProps extends IWithClassName, IWithQaAttributes {
    trip: TTripListItem;
    size: 'm' | 'l';
}

const ActiveTripItem: FC<IActiveTripItemProps> = props => {
    const {className, trip, size} = props;

    const handleTripClick = useReachGoal(
        ETripsGoal.TRIPS_PAGE_ACTIVE_TRIP_CLICK,
    );

    return (
        <Flex
            className={cx('root', `root_size_${size}`, className)}
            flexWrap="nowrap"
            {...prepareQaAttributes(props)}
        >
            <TripImage
                className={cx('image')}
                src={trip.image}
                type={ETripImageType.AUTO_HEIGHT_SQUARE}
                {...prepareQaAttributes({parent: props, current: 'tripImage'})}
            />

            <Flex className={cx('info')} between={1} flexDirection="column">
                <TripBadges trip={trip} />

                <Heading
                    className={cx('title')}
                    level="3"
                    size={size === 'm' ? 'l' : 'xxl'}
                    weight="bold"
                    {...prepareQaAttributes({parent: props, current: 'title'})}
                >
                    {trip.title}
                </Heading>

                <Text
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'displayDate',
                    })}
                >
                    {trip.displayDate}
                </Text>
            </Flex>

            <Link
                className={cx('link')}
                to={getTripOrOrderUrl(trip)}
                onClick={handleTripClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'link',
                })}
            />
        </Flex>
    );
};

export default memo(ActiveTripItem);
