import React, {useCallback} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import TTripListItem from 'types/trips/TTripListItem/TTripListItem';
import {IWithClassName} from 'types/withClassName';
import ETripImageType from 'projects/trips/components/TripImage/types/ETripImageType';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {getTripOrOrderUrl} from 'projects/account/utilities/urls/getTripOrOrderUrl';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';

import TripImage from 'projects/trips/components/TripImage/TripImage';
import Link from 'components/Link/Link';
import TripBadges from 'projects/account/pages/TripsPage/components/TripBadges/TripBadges';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './ActiveTripItem.scss';

interface IActiveTripItemProps extends IWithClassName, IWithQaAttributes {
    trip: TTripListItem;
}

const ActiveTripItem: React.FC<IActiveTripItemProps> = props => {
    const {className, trip} = props;
    const coordinator = useCoordinator();

    const handleTripClick = useCallback(() => {
        reachGoal(ETripsGoal.TRIPS_PAGE_ACTIVE_TRIP_CLICK);
        coordinator.doAction(EAppActions.REPORT_METRICS_EVENT, {
            name: ETripsGoal.TRIPS_PAGE_ACTIVE_TRIP_CLICK,
        });
    }, [coordinator]);

    const to = getTripOrOrderUrl(trip);

    return (
        <div className={cx(className, 'root')} {...prepareQaAttributes(props)}>
            <TripImage
                src={trip.image}
                type={ETripImageType.LARGE_SQUARE}
                {...prepareQaAttributes({parent: props, current: 'tripImage'})}
            />
            <TripBadges className={cx('badges')} trip={trip} />
            <div className={cx('info')}>
                <Heading
                    level={2}
                    {...prepareQaAttributes({parent: props, current: 'title'})}
                >
                    {trip.title}
                </Heading>
                <Text
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'displayDate',
                    })}
                >
                    {trip.displayDate}
                </Text>
            </div>
            <Link
                className={cx('link')}
                to={to}
                onClick={handleTripClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'link',
                })}
            />
        </div>
    );
};

export default ActiveTripItem;
