import React, {useCallback} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IWithClassName} from 'types/withClassName';
import TTripListItem from 'types/trips/TTripListItem/TTripListItem';
import ETripImageType from 'projects/trips/components/TripImage/types/ETripImageType';
import ETripState from 'types/trips/ETripState';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getTripOrOrderUrl} from 'projects/account/utilities/urls/getTripOrOrderUrl';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18n from 'i18n/account-TripsPage';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import TripImage from 'projects/trips/components/TripImage/TripImage';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './PastTripItem.scss';

interface IPastTripItemProps extends IWithClassName, IWithQaAttributes {
    trip: TTripListItem;
}

const PastTripItem: React.FC<IPastTripItemProps> = props => {
    const {className, trip} = props;

    const isCancelled = trip.state === ETripState.CANCELLED;

    const deviceType = useDeviceType();
    const to = getTripOrOrderUrl(trip);
    const coordinator = useCoordinator();

    const handleTripClick = useCallback(() => {
        coordinator.doAction(EAppActions.REPORT_METRICS_EVENT, {
            name: ETripsGoal.TRIPS_PAGE_PAST_TRIP_CLICK,
        });
    }, [coordinator]);

    return (
        <Link
            className={cx(
                className,
                'root',
                {root_cancelled: isCancelled},
                deviceMods('root', deviceType),
            )}
            to={to}
            onClick={handleTripClick}
            {...prepareQaAttributes(props)}
        >
            <Flex between={4} inline alignItems="center" nowrap>
                <TripImage
                    src={trip.image}
                    type={ETripImageType.SMALL_SQUARE}
                    enableGray={isCancelled}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'tripImage',
                    })}
                />
                <Flex className={cx('info')} flexDirection="column">
                    <Text
                        className={cx('title')}
                        size="m"
                        weight="bold"
                        overflow="ellipsis"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'title',
                        })}
                    >
                        {trip.title}
                    </Text>
                    <Text
                        className={cx('displayDate')}
                        size="m"
                        overflow="ellipsis"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'displayDate',
                        })}
                    >
                        {trip.displayDate}
                    </Text>
                    {trip.state === ETripState.CANCELLED && (
                        <Text
                            size="m"
                            color="primary"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'tripCancelled',
                            })}
                        >
                            {i18n.tripCancelled()}
                        </Text>
                    )}
                </Flex>
            </Flex>
        </Link>
    );
};

export default PastTripItem;
