import React, {useCallback, useMemo} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';
import {URLs} from 'constants/urls';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import goBack from 'utilities/browser/goBack';

import * as i18n from 'i18n/account-TripsPage';

import AccountModal from 'projects/account/components/AccountModal/AccountModal';
import ToursOrderSearchForm from 'projects/account/components/ToursOrderSearchForm/ToursOrderSearchForm';
import AuthInfo from 'projects/account/pages/TripsPage/components/SearchToursOrder/components/AuthInfo/AuthInfo';
import PlusInfo from 'projects/account/pages/TripsPage/components/SearchToursOrder/components/PlusInfo/PlusInfo';
import {useToursOrderSearchSubmit} from 'projects/account/pages/TripsPage/components/SearchToursOrder/hooks/useToursOrderSearchSubmit';
import FullscreenSpinner from 'components/FullscreenSpinner/FullscreenSpinner';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './SearchToursOrder.scss';

const ROOT_QA = 'searchToursOrderModal';

interface ISearchToursOrderProps {}

const SearchToursOrder: React.FC<ISearchToursOrderProps> = () => {
    const deviceType = useDeviceType();
    const userInfo = useUserInfo();
    const coordinator = useCoordinator();

    const handleClose = useCallback(() => {
        const processed = coordinator.doAction(
            EAppActions.NAVIGATE_TO_TRIPS_PAGE,
        );

        if (processed) {
            return;
        }

        goBack(URLs.accountTrips);
    }, [coordinator]);

    const {handleSubmit, data, isLoading, resetForm} =
        useToursOrderSearchSubmit();

    const content = useMemo(() => {
        if (!isAuthUser(userInfo)) {
            return (
                <AuthInfo className={cx('authInfo')} deviceType={deviceType} />
            );
        }

        if (!userInfo.havePlus) {
            return (
                <PlusInfo className={cx('plusInfo')} deviceType={deviceType} />
            );
        }

        return (
            <ToursOrderSearchForm
                className={cx('searchForm')}
                deviceType={deviceType}
                handleSubmit={handleSubmit}
                resetForm={resetForm}
                data={data}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'orderToursSearchForm',
                })}
            />
        );
    }, [data, deviceType, handleSubmit, resetForm, userInfo]);

    return (
        <AccountModal
            isVisible
            onClose={handleClose}
            autoclosable={deviceType.isDesktop}
            contentClassName={cx('content', deviceMods('content', deviceType))}
            backButtonTitle={i18n.searchOrderBackLink()}
            allowFullScreen
            isOwnUrlForModal
            verticalAlign={deviceType.isDesktop ? 'top-with-offset' : undefined}
            {...prepareQaAttributes(ROOT_QA)}
        >
            {content}
            {isLoading && <FullscreenSpinner />}
        </AccountModal>
    );
};

export default SearchToursOrder;
