import moment from 'moment';
import {useCallback, useState} from 'react';

import {
    EGroupName,
    IToursOrderSearchFormValues,
} from 'projects/account/components/ToursOrderSearchForm/types';
import {
    TBindToursApiResponse,
    EBindingCreationStatus,
} from 'server/api/ToursApi/types/TBindToursApiResponse';

import {useAsyncState} from 'utilities/hooks/useAsyncState';
import {ROBOT} from 'utilities/dateUtils/formats';

import {browserToursProvider} from 'serviceProvider/tours/toursProvider/browserToursProvider';

export interface IToursOrderSearchSubmitState {
    isLoading: boolean;
    data: TBindToursApiResponse | null;
    handleSubmit: (values: IToursOrderSearchFormValues) => Promise<void>;
    resetForm: () => void;
}

export function useToursOrderSearchSubmit(): IToursOrderSearchSubmitState {
    const {loading, isLoading, success, error, reset} = useAsyncState();
    const [data, setData] = useState<TBindToursApiResponse | null>(null);

    const resetForm = useCallback(() => {
        reset();
        setData(null);
    }, [reset]);

    const handleSubmit = useCallback(
        async (values: IToursOrderSearchFormValues): Promise<void> => {
            try {
                loading();

                const res = await browserToursProvider.bindTour({
                    tourId: values[EGroupName.TOURS_ORDER_SEARCH]?.tourId || '',
                    checkinDate:
                        moment(
                            values[EGroupName.TOURS_ORDER_SEARCH]?.dates
                                .startDate,
                        ).format(ROBOT) || '',
                    checkoutDate:
                        moment(
                            values[EGroupName.TOURS_ORDER_SEARCH]?.dates
                                .endDate,
                        ).format(ROBOT) || '',
                });

                success();
                setData(res);
            } catch (e) {
                error();
                setData({
                    creationStatus: EBindingCreationStatus.ERROR,
                });
            }
        },
        [error, loading, success],
    );

    return {
        isLoading,
        data,
        handleSubmit,
        resetForm,
    };
}
