import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import TTripListItem from 'types/trips/TTripListItem/TTripListItem';
import ETripNotificationType from 'types/trips/ITripNotifications/ETripNotificationType';
import isRealTripListItem from 'types/trips/TTripListItem/utilities/isRealTripListItem';
import {isNotUndefined} from 'types/utilities';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18n from 'i18n/trips-notifications';

import Label, {ELabelThemeType} from 'components/Label/Label';
import Flex from 'components/Flex/Flex';

import cx from './TripBadges.scss';

interface ITripBadgesProps extends IWithClassName {
    trip: TTripListItem;
}

const TripBadges: React.FC<ITripBadgesProps> = props => {
    const {className, trip} = props;

    const deviceType = useDeviceType();

    const labels = useMemo(() => {
        if (!isRealTripListItem(trip)) {
            return [];
        }

        const {notifications} = trip;

        if (!notifications) {
            return [];
        }

        return notifications
            .map(notification => {
                switch (notification.type) {
                    case ETripNotificationType.AVIA_CHECK_IN:
                        return {
                            key: ETripNotificationType.AVIA_CHECK_IN,
                            theme: ELabelThemeType.PRIMARY,
                            text: i18n.aviaCheckInBadgeTitle(),
                        };
                    case ETripNotificationType.HOTEL_DEFERRED_PAYMENT:
                        return {
                            key: ETripNotificationType.HOTEL_DEFERRED_PAYMENT,
                            theme: ELabelThemeType.ATTENTION,
                            text: i18n.hotelDeferredPaymentBadgeTitle(),
                        };
                    case ETripNotificationType.TRIP_CREATED:
                    case ETripNotificationType.TRIP_UPDATED:
                    default:
                        return undefined;
                }
            })
            .filter(isNotUndefined);
    }, [trip]);

    return (
        <Flex className={className} inline between={1} flexWrap="wrap">
            {labels.map((label, index) => {
                return (
                    <Label
                        key={label.key}
                        className={cx('badge')}
                        size={deviceType.isMobile ? 's' : 'm'}
                        theme={
                            index === 0 ? label.theme : ELabelThemeType.LIGHT
                        }
                    >
                        {label.text}
                    </Label>
                );
            })}
        </Flex>
    );
};

export default TripBadges;
