import React, {useMemo} from 'react';

import TTripListItem from 'types/trips/TTripListItem/TTripListItem';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18n from 'i18n/account-TripsPage';

import Heading from 'components/Heading/Heading';
import Separator from 'components/Separator/Separator';
import MoreTripsButton from 'projects/account/pages/TripsPage/components/MoreTripsButton/MoreTripsButton';
import SearchOrderLink from 'projects/account/pages/TripsPage/components/SearchOrderLink/SearchOrderLink';
import PastTrips from 'projects/account/pages/TripsPage/components/PastTrips/PastTrips';
import SearchToursOrderLink from 'projects/account/pages/TripsPage/components/SearchToursOrderLink/SearchToursOrderLink';
import ActiveTrips from 'projects/account/pages/TripsPage/components/ActiveTrips/ActiveTrips';
import NoAuthTripsPage from 'projects/account/pages/TripsPage/components/NoAuthTripsPage/NoAuthTripsPage';
import LoadingTripsPage from 'projects/account/pages/TripsPage/components/LoadingTripsPage/LoadingTripsPage';
import Container from 'components/Layouts/Container/Container';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';

import {usePlatform} from 'contexts/PlatformContext';

import cx from './TripsPageContent.scss';

interface ITripsPageContentProps extends IWithQaAttributes {
    activeTrips: TTripListItem[];
    pastTrips: TTripListItem[];
    isAuth: boolean;
    isLoading: boolean;
}

const TripsPageContent: React.FC<ITripsPageContentProps> = props => {
    const {activeTrips, pastTrips, isAuth, isLoading} = props;
    const deviceType = useDeviceType();
    const {isWeb} = usePlatform();

    const {isMobile} = deviceType;
    const rootQa = getQa(props);

    const content = useMemo(() => {
        if (!isAuth) {
            return (
                <Container className={cx('mainContainer')}>
                    <NoAuthTripsPage
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'noAuthTripsPage',
                        })}
                    />
                </Container>
            );
        }

        if (isLoading) {
            return (
                <Container className={cx('mainContainer')}>
                    <LoadingTripsPage
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'loader',
                        })}
                    />
                </Container>
            );
        }

        return (
            <>
                <Container className={cx('mainContainer')}>
                    <Heading
                        className={cx('title')}
                        level={1}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'title',
                        })}
                    >
                        {i18n.title()}
                    </Heading>

                    <div className={cx('activeTrips')}>
                        <ActiveTrips
                            trips={activeTrips}
                            {...prepareQaAttributes(rootQa)}
                        />
                    </div>

                    {isMobile && (
                        <>
                            <Separator
                                className={cx('searchOrderLinkSeparator')}
                            />

                            <HorizontalScroller className={cx('searchLinks')}>
                                <SearchOrderLink
                                    className={cx('searchOrderLink')}
                                    {...prepareQaAttributes({
                                        parent: rootQa,
                                        current: 'searchOrderLink',
                                    })}
                                />

                                {isWeb && (
                                    <SearchToursOrderLink
                                        className={cx('searchToursOrderLink')}
                                        {...prepareQaAttributes({
                                            parent: rootQa,
                                            current: 'searchToursOrderLink',
                                        })}
                                    />
                                )}
                            </HorizontalScroller>
                        </>
                    )}
                </Container>

                {Boolean(pastTrips.length) && (
                    <Container className={cx('restTrips')}>
                        <Heading
                            level="1"
                            size={isMobile ? 'xl' : 'l'}
                            weight="bold"
                            {...prepareQaAttributes({
                                parent: rootQa,
                                current: 'pastTripsTitle',
                            })}
                        >
                            {i18n.pastTripsTitle()}
                        </Heading>

                        <PastTrips
                            className={cx('pastTrips')}
                            trips={pastTrips}
                            {...prepareQaAttributes({
                                parent: rootQa,
                                current: 'pastTrips',
                            })}
                        />

                        <div className={cx('moreTripsButtonContainer')}>
                            <MoreTripsButton
                                {...prepareQaAttributes({
                                    parent: rootQa,
                                    current: 'moreTripsButton',
                                })}
                            />
                        </div>
                    </Container>
                )}
            </>
        );
    }, [activeTrips, isAuth, isLoading, isMobile, pastTrips, rootQa]);

    return (
        <div
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(rootQa)}
        >
            {content}
        </div>
    );
};

export default TripsPageContent;
