import {FunctionComponent} from 'react';
import Helmet from 'react-helmet';

import {URLs} from 'constants/urls';
import {NAVIGATION_PROJECTS_ORDER} from 'constants/navigation';
import {EAmpPageIds} from 'projects/amp/constants/EAmpPageIds';
import {YANDEX_TRAVEL_BOT_ID} from 'components/YandexMessenger/constants';

import {IWithClassName} from 'types/withClassName';

import {getAbsoluteUrl} from 'utilities/url';
import getNavigationProjectText from 'utilities/getNavigationProjectText';

import * as i18nBlock from 'i18n/account';
import * as i18nNavBlock from 'i18n/common-navigation';

import BurgerIcon from 'icons/24/Burger';
import MessageOutlineIcon from 'icons/24/MessageOutline';
import AmpLink from 'projects/amp/components/AmpLink/AmpLink';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';

import cx from './AmpSideBar.scss';

interface IAmpSideBarProps extends IWithClassName {}

const AmpSideBar: FunctionComponent<IAmpSideBarProps> = ({className}) => {
    return (
        <>
            <Helmet>
                <script
                    async
                    custom-element="amp-sidebar"
                    src="https://cdn.ampproject.org/v0/amp-sidebar-0.1.js"
                />
            </Helmet>

            <button
                className={cx('button', className)}
                // @ts-ignore
                on={`tap:${EAmpPageIds.NAV_SIDEBAR}.toggle`}
            >
                <BurgerIcon />
            </button>

            <amp-sidebar
                id={EAmpPageIds.NAV_SIDEBAR}
                class={cx('sideBar')}
                layout="nodisplay"
                side="right"
            >
                <nav>
                    <ul className={cx('navList')}>
                        {NAVIGATION_PROJECTS_ORDER.map(project => (
                            <li key={project}>
                                <AmpLink
                                    className={cx('navLink')}
                                    url={getAbsoluteUrl(URLs[project])}
                                    theme="black"
                                >
                                    {getNavigationProjectText(project)}
                                </AmpLink>
                            </li>
                        ))}
                        <li className={cx('navItem_with_separator')}>
                            <AmpLink
                                className={cx('navLink')}
                                theme="black"
                                url={getAbsoluteUrl(URLs.promoCertificatesUrl)}
                            >
                                {i18nNavBlock.certificates()}
                            </AmpLink>
                        </li>
                        <li className={cx('navItem_with_separator')}>
                            <AmpLink
                                className={cx('navLink')}
                                url={getAbsoluteUrl(URLs.accountTrips)}
                                theme="black"
                            >
                                {i18nBlock.menuDotTrips()}
                            </AmpLink>
                        </li>
                        <li>
                            <AmpLink
                                className={cx('navLink')}
                                url={`https://yandex.ru/chat/#/user/${YANDEX_TRAVEL_BOT_ID}`}
                                theme="black"
                                target="_blank"
                            >
                                <TextWithIcon
                                    size="m"
                                    iconSize={24}
                                    iconLeft={MessageOutlineIcon}
                                    text={i18nBlock.chatWithSupport()}
                                />
                            </AmpLink>
                        </li>
                    </ul>
                </nav>
            </amp-sidebar>
        </>
    );
};

export default AmpSideBar;
