import {FunctionComponent} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IHotel} from 'types/hotels/hotel/IHotel';

import {convertHotelOrRoomImages} from 'projects/hotels/utilities/prepareAndParseImages/prepareAndParseImages';

import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import Separator from 'components/Separator/Separator';
import HotelStars from 'components/HotelStars/HotelStars';
import AmpImage from 'projects/amp/components/AmpImage/AmpImage';
import HotelName from 'projects/hotels/components/HotelName/HotelName';
import HotelAmenities from './components/HotelAmenities/HotelAmenities';
import AmpCarousel from 'projects/amp/components/AmpCarousel/AmpCarousel';
import AmpHotelAddress from './components/AmpHotelAddress/AmpHotelAddress';
import HotelGeoFeature from 'projects/hotels/components/HotelGeoFeature/HotelGeoFeature';
import AmpHotelRating from 'projects/amp/pages/AmpHotelPage/components/AmpHotelRating/AmpHotelRating';
import TransportAccessibility from 'projects/hotels/components/TransportAccessibility/TransportAccessibility';

import cx from './HotelCard.scss';

const HOTEL_IMAGE_SIZE_NAME = 'L';
const CAROUSEL_HEIGHT = 200;

interface IHotelCardProps extends IWithClassName {
    hotel: IHotel;
}

const HotelCard: FunctionComponent<IHotelCardProps> = ({className, hotel}) => {
    const {
        name,
        stars,
        images,
        geoFeature,
        rating,
        totalTextReviewCount,
        nearestStations,
    } = hotel;
    const convertedImages = convertHotelOrRoomImages(
        images,
        HOTEL_IMAGE_SIZE_NAME,
    ).map(image => ({...image, alt: name}));

    return (
        <Card className={className} inset="3">
            <Heading level="1" size="xl">
                <HotelName
                    hotelName={name}
                    icon={<HotelStars stars={stars} size="16" />}
                />
            </Heading>

            <AmpHotelRating
                className={cx('reviews')}
                rating={rating}
                totalTextReviewCount={totalTextReviewCount}
                hotelSlug={hotel.hotelSlug}
            />

            <AmpCarousel
                className={cx('imagesCarousel')}
                height={CAROUSEL_HEIGHT}
            >
                {convertedImages.map(image => {
                    if (!image.size) {
                        return null;
                    }

                    const width =
                        (CAROUSEL_HEIGHT / image.size?.height) *
                        image.size?.width;

                    return (
                        image.size && (
                            <AmpImage
                                key={image.src}
                                src={image.src}
                                alt={image.alt}
                                responsiveSet={image.additionalSizes}
                                height={CAROUSEL_HEIGHT}
                                width={width}
                                withSchemaOrg
                            >
                                <meta itemProp="name" content={name} />
                                <meta itemProp="url" content={image.src} />
                                <meta
                                    itemProp="width"
                                    content={width.toString()}
                                />
                                <meta
                                    itemProp="height"
                                    content={CAROUSEL_HEIGHT.toString()}
                                />
                            </AmpImage>
                        )
                    );
                })}
            </AmpCarousel>

            <Separator margin="4" />

            <AmpHotelAddress hotel={hotel} />

            <Flex between="4" above="4" inline>
                <HotelGeoFeature geoFeature={geoFeature} />
                {nearestStations?.[0] && (
                    <TransportAccessibility
                        bigIcon
                        station={nearestStations[0]}
                    />
                )}
            </Flex>

            <HotelAmenities
                className={cx('amenities')}
                mainAmenities={hotel.mainAmenities}
                amenityGroups={hotel.amenityGroups}
            />
        </Card>
    );
};

export default HotelCard;
