import React, {useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {Route, Switch, useLocation} from 'react-router-dom';

import {URLs} from 'constants/urls';

import {ELoadableReducer} from 'types/common/ELoadableReducer';

import {currenciesFetchActions} from 'reducers/common/currencies/actions';
import {resetFilterValues} from 'reducers/avia/search/results/filters/actions';
import pageReducer from 'reducers/avia/page/reducer';
import contextReducer from 'reducers/avia/context/reducer';
import calendarPricesReducer from 'reducers/avia/aviaCalendarPrices/reducer';
import bookingReducer from 'reducers/avia/booking/reducer';
import searchReducer from 'reducers/avia/search/reducer';
import orderReducer from 'reducers/avia/order/reducer';
import pointsDataReducer from 'reducers/avia/pointsData/reducer';
import redirectReducer from 'reducers/avia/redirect/reducer';
import priceIndexReducer from 'reducers/avia/aviaPriceIndex/reducer';
import routeReducer from 'reducers/avia/route/reducer';
import flightsToReducer from 'reducers/avia/flightsTo/reducer';
import countryRestrictions from 'reducers/avia/countryRestrictions/reducer';
import weatherForecast from 'reducers/avia/weatherForecast/reducer';
import aviaLogMetricsReducer from 'reducers/avia/aviaLogMetrics/reducer';
import happyPageReducer from 'reducers/happyPage/reducer';
import searchSuggestsReducer from 'reducers/avia/searchSuggests/reducer';
import personalizationReducer from 'reducers/avia/personalization/reducer';

import {aviaOrderSelectors} from 'selectors/avia/order/aviaOrderSelector';

import aviaSaga from 'sagas/avia/rootSaga';
import pricesSaga from 'sagas/avia/aviaCalendarPrices/aviaCalendarPricesSaga';
import requestHappyPageSaga from 'sagas/happyPage/orderAndCrossSales/requestHappyPageSaga';

import {loadable} from 'utilities/pageLoadable';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {IBrowserHistoryState} from 'utilities/browserHistory/browserHistory';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import AppRoute from 'components/AppRoute/AppRoute';
import RouteMetaHelmet from 'projects/avia/pages/AviaLanding/components/RouteMetaHelmet/RouteMetaHelmet';
import FlightsToMetaHelmet from 'projects/avia/pages/AviaLanding/components/FlightsToMetaHelmet/FlightsToMetaHelmet';

import Subscription from 'projects/avia/pages/AviaSubscription';
import {
    ModalOrderRouted,
    OrderAsPage,
} from 'projects/avia/pages/AviaOrder/AviaOrderPage';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import {fetchCurrencies} from 'server/redux/common/fetchCurrencies';

import {AviaSearchMeta} from './containers/Meta';

const SearchResult = loadable(() => import('./pages/AviaSearchResult/index'));

const InvalidSearch = loadable(
    () => import('./pages/InvalidSearch/InvalidSearch'),
);

const Booking = loadable(() => import('./pages/AviaBooking/AviaBooking'));

const BookingOrderPage = loadable(() => import('./pages/AviaBookingOrder'));

const NotFoundApp = loadable(
    () => import('components/NotFound/NotFoundPage/NotFoundPage'),
);

const AviaFlightsTo = loadable(
    () => import('./pages/AviaLanding/AviaFlightsTo/AviaFlightsTo'),
);

const Redirect = loadable(() => import('./pages/AviaRedirect/AviaRedirect'));

const RedirectError = loadable(
    () => import('./pages/AviaRedirectError/AviaRedirectError'),
);

const RoutePage = loadable(
    () => import('./pages/AviaLanding/AviaRoute/AviaRoute'),
);

const AviaSearchToCountryPage = loadable(
    () => import('./pages/AviaSearchToCountry/AviaSearchToCountryPage'),
);

const AviaSearchToAnywherePage = loadable(
    () => import('./pages/AviaSearchToAnywhere/AviaSearchToAnywherePage'),
);

const FlightPage = loadable(() => import('./pages/FlightPage/FlightPage'));

const AviaApp: React.FC = () => {
    const dispatch = useDispatch();
    const {isIe} = useDeviceType();
    const useOrderData = useSelector(aviaOrderSelectors.useOrderData);
    const location = useLocation<IBrowserHistoryState>();
    const shouldUseBackground = !isIe && !useOrderData;
    const background =
        shouldUseBackground && location.state && location.state.background;

    useEffect(() => {
        dispatch(currenciesFetchActions.request());

        return (): void => {
            dispatch(resetFilterValues());
        };
    }, [dispatch]);

    return (
        <>
            {background && (
                <Route path={URLs.aviaOrder} component={ModalOrderRouted} />
            )}

            <Switch location={background || location}>
                <Route path={URLs.aviaSearchError} component={InvalidSearch} />

                <AppRoute
                    path={URLs.aviaSearchResults}
                    render={(props): React.ReactNode => (
                        <SearchResult
                            {...props}
                            isBackground={Boolean(background)}
                        />
                    )}
                    meta={AviaSearchMeta}
                />

                <Route path={URLs.aviaOrder} component={OrderAsPage} />

                <Route path={URLs.aviaSubscription} component={Subscription} />

                <Route path={URLs.aviaBooking} component={Booking} />

                <Route path={URLs.aviaOrderById} component={BookingOrderPage} />

                <Route
                    path={URLs.aviaRedirectError}
                    component={RedirectError}
                />

                <Route path={URLs.aviaRedirect} component={Redirect} />

                <AppRoute
                    path={URLs.aviaRoute}
                    component={RoutePage}
                    meta={RouteMetaHelmet}
                />

                <AppRoute
                    path={URLs.aviaFlightsTo}
                    component={AviaFlightsTo}
                    meta={FlightsToMetaHelmet}
                />

                <Route path={URLs.aviaFlight} component={FlightPage} />

                <Route
                    path={URLs.aviaSearchToCountry}
                    component={AviaSearchToCountryPage}
                />

                <Route
                    path={URLs.aviaSearchToAnywhere}
                    component={AviaSearchToAnywherePage}
                />

                <Route component={NotFoundApp} />
            </Switch>
        </>
    );
};

export default withReducers([
    [ELoadableReducer.AVIA_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.AVIA_PAGE, pageReducer],
    [ELoadableReducer.AVIA_CONTEXT, contextReducer],
    [ELoadableReducer.AVIA_CALENDAR_PRICES, calendarPricesReducer],
    [ELoadableReducer.AVIA_BOOKING, bookingReducer],
    [ELoadableReducer.AVIA_SEARCH, searchReducer],
    [ELoadableReducer.AVIA_ORDER, orderReducer],
    [ELoadableReducer.AVIA_POINTS_DATA, pointsDataReducer],
    [ELoadableReducer.AVIA_REDIRECT, redirectReducer],
    [ELoadableReducer.AVIA_PRICE_INDEX, priceIndexReducer],
    [ELoadableReducer.AVIA_ROUTE, routeReducer],
    [ELoadableReducer.AVIA_FLIGHTS_TO, flightsToReducer],
    [ELoadableReducer.AVIA_COUNTRY_RESTRICTIONS, countryRestrictions],
    [ELoadableReducer.AVIA_WEATHER_FORECAST, weatherForecast],
    [ELoadableReducer.HAPPY_PAGE, happyPageReducer],
    [ELoadableReducer.AVIA_LOG_METRICS, aviaLogMetricsReducer],
    [ELoadableReducer.AVIA_PERSONALIZATION, personalizationReducer],
])(
    withSaga([aviaSaga, pricesSaga, requestHappyPageSaga])(
        serverFetchDataDispatcher([fetchCurrencies], {checkNested: true})(
            AviaApp,
        ),
    ),
);
