import React from 'react';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {TIME, HUMAN_SHORT} from 'utilities/dateUtils/formats';
import {parseDate} from 'utilities/dateUtils';
import {useMobile} from 'utilities/hooks/useMobile';

import * as i18nBlock from 'i18n/avia-AviaBooking-BookingFlightInfo';

import {Duration} from 'projects/avia/components/Duration/Duration';

import cx from './AviaFlightTimings.scss';

interface IFlightTimingsProps extends IWithQaAttributes {
    departureDate: string;
    arrivalDate: string;
    departureAirportTimezoneOffset: number;
    arrivalAirportTimezoneOffset: number;
    /**
     * Показывать дату рядом со временем
     */
    showDate?: boolean;

    view?: 'search' | 'booking' | 'order';
}

export const AviaFlightTimings: React.FC<IFlightTimingsProps> = props => {
    const {
        departureDate,
        arrivalDate,
        arrivalAirportTimezoneOffset,
        departureAirportTimezoneOffset,
        showDate = false,
        view,
    } = props;
    const isMobile = useMobile();
    const parsedDepartureDate = parseDate(departureDate);
    const parsedArrivalDate = parseDate(arrivalDate);
    const departureDateTime = parsedDepartureDate.utcOffset(
        departureAirportTimezoneOffset,
        true,
    );
    const arrivalDateTime = parsedArrivalDate.utcOffset(
        arrivalAirportTimezoneOffset,
        true,
    );
    const diffInMinutes = arrivalDateTime.diff(departureDateTime, 'minutes');

    const parsedArrivalStartOfDay = parsedArrivalDate
        .clone()
        .startOf('day')
        .utcOffset(0);
    const parsedDepartureStartOfDay = parsedDepartureDate
        .clone()
        .startOf('day')
        .utcOffset(0);

    const dayIsChanged = parsedArrivalStartOfDay.isAfter(
        parsedDepartureStartOfDay,
        'day',
    ); // проверка что день изменился

    const twoDaysChanged =
        dayIsChanged &&
        parsedArrivalDate.isAfter(
            parsedDepartureStartOfDay.clone().add(1, 'day'),
            'day',
        ); // проверка что день изменился больше чем на 1

    const datesShowed = showDate || dayIsChanged;

    return (
        <div
            className={cx(
                'root',
                view && `root_view-${view}`,
                isMobile && 'root_mobile',
                datesShowed && 'root_day-changed',
            )}
        >
            {datesShowed && (
                <div className={cx('dates')}>
                    {showDate && (
                        <div className={cx('date')}>
                            {parsedDepartureDate.format(HUMAN_SHORT)}
                        </div>
                    )}
                    <div className={cx('date')}>
                        {view === 'search' || twoDaysChanged || !dayIsChanged
                            ? parsedArrivalDate.format(HUMAN_SHORT)
                            : i18nBlock.nextDashDay({
                                  date: parsedArrivalDate.format(HUMAN_SHORT),
                              })}
                    </div>
                </div>
            )}
            <span
                className={cx('time')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'departure',
                })}
            >
                {parsedDepartureDate.format(TIME)}
            </span>
            <span className={cx('duration')}>
                <Duration
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'duration',
                    })}
                    duration={Duration.minutes(diffInMinutes)}
                    view={view}
                />
            </span>
            <span
                className={cx('time')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'arrival',
                })}
            >
                {parsedArrivalDate.format(TIME)}
            </span>
        </div>
    );
};
