import {FC, useEffect, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {MAX_DAYS_RANGE} from 'constants/hotels';

import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';
import {EYtpReferer} from 'types/common/IAttribution';
import {IWithClassName} from 'types/withClassName';

import {aviaContextForCrossSaleSelector} from 'selectors/avia/search/aviaContextForCrossSaleSelector';

import {isV3Response} from 'utilities/crossSaleHotels/isV3Response';
import {parseDate} from 'utilities/dateUtils';
import getFormattedDatesForPeriod from 'projects/avia/utilities/getFormattedDatesForPeriod';
import {ROBOT} from 'utilities/dateUtils/formats';
import {getCrossSaleHotelsSearchUrl} from 'projects/avia/utilities/getCrossSaleHotelsSearchUrl';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/avia-crossSale';

import useHotelsCrossSale from 'hooks/useHotelsCrossSale/useHotelsCrossSale';

import HotelsCrossSaleMap from 'components/HotelsCrossSaleMap/HotelsCrossSaleMap';
import Card from 'components/Card/Card';
import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';
import Flex from 'components/Flex/Flex';
import IconWithRoundBackground from 'components/IconWithRoundBackground/IconWithRoundBackground';
import HotelIcon from 'icons/16/Hotel';
import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import AviaHotelsCrossSaleMapSidebarSkeleton from 'projects/avia/components/AviaHotelsCrossSaleMapSidebar/components/AviaHotelsCrossSaleMapSidebarSkeleton/AviaHotelsCrossSaleMapSidebarSkeleton';

import cx from './AviaHotelsCrossSaleMapSidebar.scss';

export interface IAviaHotelsCrossSaleMapSidebarProps extends IWithClassName {}

const AviaHotelsCrossSaleMapSidebar: FC<IAviaHotelsCrossSaleMapSidebarProps> =
    ({className}) => {
        const requestParams = useSelector(aviaContextForCrossSaleSelector);
        const {fetch, reset, isError, isLoading, data} = useHotelsCrossSale();

        const checkinDateMoment = useMemo(() => {
            if (!requestParams) {
                return undefined;
            }

            return parseDate(requestParams?.checkinDate);
        }, [requestParams]);
        const checkoutDateMoment = useMemo(() => {
            if (!requestParams || !checkinDateMoment) {
                return undefined;
            }

            return requestParams?.checkoutDate
                ? parseDate(requestParams.checkoutDate)
                : checkinDateMoment.clone().add(1, 'day');
        }, [checkinDateMoment, requestParams]);

        const correctCrossSaleParams =
            checkinDateMoment && checkoutDateMoment
                ? !checkinDateMoment.isSame(checkoutDateMoment) &&
                  checkoutDateMoment.diff(checkinDateMoment, 'days') <=
                      MAX_DAYS_RANGE
                : false;

        const crossSale = useMemo(() => {
            if (isError || isLoading) {
                return null;
            }

            return data && (isV3Response(data) ? data.crossSale : data);
        }, [data, isLoading, isError]);

        useEffect(() => {
            if (!correctCrossSaleParams || !requestParams) {
                return;
            }

            fetch(requestParams);

            return (): void => {
                reset();
            };
        }, [fetch, requestParams, reset, correctCrossSaleParams]);

        const offerSearchParams: Partial<IRequiredOfferParams> = useMemo(() => {
            return {
                checkinDate: checkinDateMoment?.format(ROBOT),
                checkoutDate: checkoutDateMoment?.format(ROBOT),
                adults: requestParams?.adults,
                childrenAges: requestParams?.childrenAges,
            };
        }, [
            checkinDateMoment,
            checkoutDateMoment,
            requestParams?.adults,
            requestParams?.childrenAges,
        ]);

        const additionalTitle = useMemo(() => {
            if (!crossSale || !checkinDateMoment || !checkoutDateMoment) {
                return;
            }

            const [checkinDateText, checkoutDateText] =
                getFormattedDatesForPeriod(
                    checkinDateMoment,
                    checkoutDateMoment,
                );

            return i18nBlock.dates({
                checkinDate: checkinDateText,
                checkoutDate: checkoutDateText,
            });
        }, [checkinDateMoment, checkoutDateMoment, crossSale]);

        if (!correctCrossSaleParams || !requestParams) {
            return null;
        }

        if (isLoading || (!isError && !crossSale)) {
            return <AviaHotelsCrossSaleMapSidebarSkeleton />;
        }

        if (!crossSale) {
            return null;
        }

        const url = getCrossSaleHotelsSearchUrl({
            geoId: crossSale.region.geoId,
            offerSearchParams,
            ytpReferer: EYtpReferer.AVIA_SEARCH_MAP,
        });

        return (
            <Card shadow="default" className={cx(className)}>
                <Box x={6} y={4}>
                    <Heading level={3}>
                        {i18nBlock.hotelsIn({
                            regionName:
                                crossSale.region?.linguistics.prepositionalCase,
                        })}
                    </Heading>
                    <Flex y={3} alignItems="center">
                        <IconWithRoundBackground
                            icon={<HotelIcon className={cx('hotelIcon')} />}
                            backgroundColor="label-secondary"
                            className={cx('icon')}
                        />

                        <Link
                            to={url}
                            className={cx('variantsLink')}
                            target="_blank"
                            theme="black"
                        >
                            <Text size="m">
                                {insertJSXIntoKey(
                                    i18nBlock.sidebarVariantsCount,
                                )({
                                    hotelsCount: crossSale.totalHotelCount,
                                    price: (
                                        <Price
                                            {...crossSale.minPriceInRegion}
                                        />
                                    ),
                                    additionalTitle: additionalTitle ?? '',
                                })}
                            </Text>
                        </Link>
                    </Flex>
                    <HotelsCrossSaleMap
                        className={cx('map')}
                        crossSaleData={crossSale}
                        linkUrl={url}
                        hideActiveHotel
                    />
                    <Flex above={3}>
                        <Text size="m">
                            <Link
                                to={url}
                                className={cx('selectHotelLink')}
                                target="_blank"
                            >
                                {i18nBlock.selectHotel()}
                            </Link>
                        </Text>
                    </Flex>
                </Box>
            </Card>
        );
    };

export default AviaHotelsCrossSaleMapSidebar;
