import React, {PureComponent} from 'react';
import {RouteComponentProps, withRouter} from 'react-router-dom';
import {renderToString} from 'react-dom/server';
import {IYMapsApi} from 'react-yandex-maps';
import {connect} from 'react-redux';

import {IToPoint} from 'server/services/AviaNearService/types/IAviaNear';
import {IPrepareNearItem} from 'projects/avia/components/MapNearAirports/types';

import {StoreInterface} from 'reducers/storeTypes';

import {currenciesConverterSelector} from 'selectors/common/currenciesSelector';

import {PriceConverter} from 'utilities/currency/priceConverter';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/avia-NearAirports';

import {YandexMapPlacemark} from 'components/YandexMaps/YandexMaps';
import PriceWithoutConvertor from 'components/PriceWithoutConvertor/PriceWithoutConvertor';
import LocationIcon from 'icons/36/Location';

import cx from './AviaMapMarker.scss';

interface IAviaMapCommonMarkerProps
    extends IAviaMapMarkerStateProps,
        RouteComponentProps {
    className?: string;
    yandexMapAPIInstance: IYMapsApi | null;
}

interface IAviaMapMarkerStateProps {
    priceConverter: PriceConverter;
}

interface IAviaMapCenterMarkerProps extends IAviaMapCommonMarkerProps {
    item: IToPoint;
    center: true;
}

interface IAviaMapNearMarkerProps extends IAviaMapCommonMarkerProps {
    item: IPrepareNearItem;
    center?: false;
}

type TAviaMapMarkerProps = IAviaMapCenterMarkerProps | IAviaMapNearMarkerProps;

class AviaMapMarker extends PureComponent<TAviaMapMarkerProps> {
    private renderAviaMarker = (): React.ReactElement => {
        return (
            <div className={cx('root', {root_center: this.props.center})}>
                <div className={cx('container')} data-type-element="container">
                    <div className={cx('balloon')} data-type-element="balloon">
                        <div className={cx('balloon__text')}>
                            <div className={cx('text__first-line')}>
                                {this.props.center
                                    ? this.props.item.name
                                    : this.props.item.title}
                            </div>
                            {this.renderSecondLine()}
                        </div>
                    </div>
                    <LocationIcon
                        className={cx('pin')}
                        width={38}
                        height={38}
                    />
                </div>
            </div>
        );
    };

    private renderSecondLine(): React.ReactNode {
        if (this.props.center) {
            return null;
        }

        const price = this.props.item.price && (
            <PriceWithoutConvertor
                {...this.props.item.price}
                isFrom
                className={cx('text__price')}
                priceConverter={this.props.priceConverter}
            />
        );
        const distance = i18nBlock.km({
            distance: Math.round(this.props.item.distance),
            city: '',
        });

        return (
            <div className={cx('text__theme_secondary')}>
                {insertJSXIntoKey(i18nBlock.priceDashSepDashDistance)({
                    price,
                    distance,
                })}
            </div>
        );
    }

    private handleMarkerClick = (): void => {
        if (this.props.center) {
            return;
        }

        this.props.history.push(this.props.item.url);
    };

    render(): React.ReactNode {
        const {
            yandexMapAPIInstance,
            item: {
                point: [lat, lng],
            },
        } = this.props;

        if (!yandexMapAPIInstance) {
            return null;
        }

        const aviaMarkerNode = this.renderAviaMarker();
        const markerContentLayout =
            yandexMapAPIInstance.templateLayoutFactory.createClass(
                renderToString(aviaMarkerNode),
                {
                    build: function () {
                        this.constructor.superclass.build.call(this);
                        this.container = this.getElement().querySelector(
                            '[data-type-element="container"]',
                        );
                        this.balloon = this.getElement().querySelector(
                            '[data-type-element="balloon"]',
                        );
                    },
                    getShape: function () {
                        const {
                            offsetWidth: markerWidth,
                            offsetHeight: markerHeight,
                        } = this.balloon;
                        const {top: topInStr, left: leftInStr} =
                            getComputedStyle(this.container);
                        const top = parseInt(topInStr || '0', 10);
                        const left = parseInt(leftInStr || '0', 10);

                        return new yandexMapAPIInstance.shape.Rectangle(
                            new yandexMapAPIInstance.geometry.pixel.Rectangle([
                                [left, top],
                                [left + markerWidth, top + markerHeight],
                            ]),
                        );
                    },
                },
            );

        return (
            <YandexMapPlacemark
                key={`${lat}-${lng}`}
                geometry={[lat, lng]}
                onClick={this.handleMarkerClick}
                options={{iconLayout: markerContentLayout}}
            />
        );
    }
}

export default withRouter(
    connect(
        (state: StoreInterface): IAviaMapMarkerStateProps => ({
            priceConverter: currenciesConverterSelector(state),
        }),
    )(AviaMapMarker),
);
