import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {useToggle} from 'utilities/hooks/useToggle';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {EAviaOfferType} from 'projects/avia/lib/search/offerType';

import * as i18nBlock from 'i18n/avia-AviaOrder-PriceList';
import * as i18nAviaOrderBlock from 'i18n/avia-order';

import Box from 'components/Box/Box';
import LinkButton from 'components/LinkButton/LinkButton';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import AviaOrderOffer from 'projects/avia/pages/AviaOrder/components/AviaOrderOffer/AviaOrderOffer';
import AviaOrderOfferListPlaceholder from 'projects/avia/pages/AviaOrder/components/OfferListPlaceholder';

import cx from './AviaPriceList.scss';

const MAX_INITIAL_OFFERS = 6;

interface IAviaPriceListProps extends IWithClassName, IWithQaAttributes {
    offerType?: EAviaOfferType;
    cheapest?: IResultAviaVariant;
    avia?: IResultAviaVariant[];
    other?: IResultAviaVariant[];
    qid?: string;
    visibleCountOffers?: number;
    onClick?: (offer: IResultAviaVariant) => void;
}

const AviaPriceList: React.FC<IAviaPriceListProps> = props => {
    const {
        className,
        offerType,
        cheapest,
        avia = [],
        other = [],
        qid,
        visibleCountOffers = MAX_INITIAL_OFFERS,
        onClick,
    } = props;
    const [moreIsShown, toggleMore] = useToggle(false);

    const cheapestCount = cheapest ? 1 : 0;
    const offersCount = cheapestCount + avia.length + other.length;

    if (!offersCount || !qid) {
        return <AviaOrderOfferListPlaceholder />;
    }

    const qa = prepareQaAttributes(props);

    const isManyOffers = offersCount - visibleCountOffers > 0;

    const aviaCompanyOffer = avia.length ? avia[0] : null;
    const isAviaBoyOffer = aviaCompanyOffer?.price.boy;

    const otherCount =
        visibleCountOffers -
        (cheapestCount + (isAviaBoyOffer ? 1 : avia.length));

    const otherList = other.slice(0, otherCount);
    const additionalList = other.slice(otherCount);

    return (
        <Box className={className} between="3" {...qa}>
            {cheapest && (
                <>
                    <div
                        className={cx('offersTitle', 'hasSeparator', className)}
                    >
                        {cheapest.price.fromCompany && !cheapest.price.boy
                            ? i18nBlock.titleDashBestDashAvia()
                            : i18nBlock.titleDashBest()}
                    </div>
                    <AviaOrderOffer
                        offerType={offerType}
                        offer={cheapest}
                        qid={qid ?? ''}
                        isBestOffer
                        onClick={onClick}
                        {...prepareQaAttributes({
                            key: cheapest.price.fromCompany
                                ? 'cheapest-company'
                                : 'cheapest',
                            parent: qa,
                            current: 'offer',
                        })}
                    />
                </>
            )}
            {isAviaBoyOffer && aviaCompanyOffer && (
                <>
                    <div
                        className={cx('offersTitle', 'hasSeparator', className)}
                    >
                        {i18nBlock.titleDashBoy()}
                    </div>
                    <AviaOrderOffer
                        offerType={offerType}
                        offer={aviaCompanyOffer}
                        qid={qid ?? ''}
                        onClick={onClick}
                        {...prepareQaAttributes({
                            key: 'boy',
                            parent: qa,
                            current: 'offer',
                        })}
                    />
                </>
            )}
            {!isAviaBoyOffer && avia.length > 0 && (
                <>
                    <div
                        className={cx('offersTitle', 'hasSeparator', className)}
                    >
                        {i18nBlock.titleDashAvia()}
                    </div>
                    {avia.map((offer, idx) => (
                        <AviaOrderOffer
                            key={'title-avia-offer-' + idx}
                            offerType={offerType}
                            offer={offer}
                            qid={qid ?? ''}
                            onClick={onClick}
                            {...prepareQaAttributes({
                                key: 'company',
                                parent: qa,
                                current: 'offer',
                            })}
                        />
                    ))}
                </>
            )}
            {otherList.length > 0 && (
                <>
                    <div
                        className={cx('offersTitle', 'hasSeparator', className)}
                    >
                        {i18nBlock.titleDashOther()}
                    </div>
                    {otherList.map((offer, idx) => (
                        <AviaOrderOffer
                            key={'title-other-' + idx}
                            offerType={offerType}
                            offer={offer}
                            qid={qid ?? ''}
                            onClick={onClick}
                            {...prepareQaAttributes({
                                key: idx,
                                parent: qa,
                                current: 'offer',
                            })}
                        />
                    ))}
                </>
            )}
            {isManyOffers && !moreIsShown && (
                <Box textSize="m" below={1}>
                    <LinkButton onClick={toggleMore}>
                        {i18nAviaOrderBlock.showDashMoreDashOffers()}
                    </LinkButton>
                </Box>
            )}
            {additionalList.length > 0 && (
                <AutoHeightAnimateControl canShowComponent={moreIsShown}>
                    <Box between="5">
                        {additionalList.map((offer, idx) => (
                            <AviaOrderOffer
                                key={'title-other-' + idx}
                                offerType={offerType}
                                offer={offer}
                                qid={qid ?? ''}
                                onClick={onClick}
                                {...prepareQaAttributes({
                                    key: idx,
                                    parent: qa,
                                    current: 'offer',
                                })}
                            />
                        ))}
                    </Box>
                </AutoHeightAnimateControl>
            )}
        </Box>
    );
};

export default React.memo(AviaPriceList);
