import React, {useEffect, useState} from 'react';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IAviaVariantGroup} from 'selectors/avia/utils/denormalization/variantGroup';

import {useToggle} from 'utilities/hooks/useToggle';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    EAviaOfferType,
    getAviaOfferType,
} from 'projects/avia/lib/search/offerType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import AviaMobileResultVariant from './components/AviaMobileResultVariant/AviaMobileResultVariant';
import AviaDesktopResultVariant from './components/AviaDesktopResultVariant/AviaDesktopResultVariant';
import useSyncSelectedVariant from 'projects/avia/components/AviaResultVariant/hooks/useSyncSelectedVariant';
import useUniqVariants from './components/TariffSelector/hooks/useUniqVariants';
import Skeleton from 'components/Skeletons/Skeleton/Skeleton';

import useSyncBaggageToggleWithGlobalBaggageFilter from './hooks/useSyncBaggageToggleWithGlobalBaggageFilter';

import cx from './AviaResultVariantContainer.scss';

export interface IAviaResultVariantProps extends IWithQaAttributes {
    group: IAviaVariantGroup;
    selectedVariant: IResultAviaVariant;
    baggageToggleIsActive: boolean;
    offerType?: EAviaOfferType;
    variants: IResultAviaVariant[];
    onVariantPriceButtonClick?: () => void;
}

interface IAviaResultVariantContainerProps {
    group: IAviaVariantGroup;
    renderIsDynamic?: boolean;
}

const AviaResultVariantContainer: React.FC<IAviaResultVariantContainerProps> =
    ({
        group,
        group: {
            cheapestVariantsByTariff: {cheapest, cheapestWithBaggage},
        },
        renderIsDynamic = false,
    }) => {
        const deviceType = useDeviceType();
        const [showSkeleton, setSkeletonVisibility] = useState(renderIsDynamic);
        const [selectedVariant, setSelectedVariant] = useState(cheapest);
        const [baggageToggleIsActive, toggleBaggage] = useToggle(
            cheapest.hasBaggage,
        );

        useEffect(() => {
            setSelectedVariant(
                (baggageToggleIsActive && cheapestWithBaggage) || cheapest,
            );
        }, [baggageToggleIsActive, cheapest, cheapestWithBaggage]);

        useEffect(() => {
            if (showSkeleton) {
                setTimeout(() => setSkeletonVisibility(false));
            }
        }, [showSkeleton]);

        useSyncBaggageToggleWithGlobalBaggageFilter(
            cheapest.hasBaggage,
            baggageToggleIsActive,
            toggleBaggage,
        );

        useSyncSelectedVariant(
            cheapest,
            cheapestWithBaggage,
            setSelectedVariant,
        );

        const variants = useUniqVariants(
            group.cheapestVariantsByTariff.withoutBaggage.noOrUnknownRefund,
            group.cheapestVariantsByTariff.withoutBaggage.chargedRefund,
            group.cheapestVariantsByTariff.withoutBaggage.freeRefund,
            group.cheapestVariantsByTariff.withBaggage.noOrUnknownRefund,
            group.cheapestVariantsByTariff.withBaggage.chargedRefund,
            group.cheapestVariantsByTariff.withBaggage.freeRefund,
        );

        const onVariantPriceButtonClick = useReachGoal(
            EAviaGoal.SEARCH_VARIANT_PRICE_BUTTON_CLICK,
        );

        if (showSkeleton) {
            return <Skeleton className={cx('skeleton')} />;
        }

        const AviaResultVariant = deviceType.isMobile
            ? AviaMobileResultVariant
            : AviaDesktopResultVariant;

        return (
            <AviaResultVariant
                group={group}
                selectedVariant={selectedVariant}
                offerType={getAviaOfferType(group.variants)}
                baggageToggleIsActive={baggageToggleIsActive}
                variants={variants}
                onVariantPriceButtonClick={onVariantPriceButtonClick}
                {...prepareQaAttributes({
                    key: group.key,
                    current: 'aviaSearchVariant',
                })}
            />
        );
    };

AviaResultVariantContainer.displayName = 'AviaResultVariantContainer';

export default AviaResultVariantContainer;
