import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {CHAR_MULTIPLICATION_SIGN} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nUnitsBlock from 'i18n/common-units';
import * as i18nBlock from 'i18n/avia-baggage-factory';

import BaggageIconWithWeight, {
    EBaggageWithWeightIconType,
} from 'components/BaggageIconWithWeight/BaggageIconWithWeight';
import Flex from 'components/Flex/Flex';
import useBaggageInfo from 'projects/avia/components/AviaResultVariant/hooks/useBaggageInfo';
import {TTextSize} from 'components/Text/Text';

import cx from './AviaBaggageInfo.scss';

interface IAviaBaggageInfoProps extends IWithClassName, IWithQaAttributes {
    textSize?: TTextSize;
    variant: IResultAviaVariant;
}

export const AviaBaggageInfo: React.FC<IAviaBaggageInfoProps> = ({
    className,
    textSize = 'm',
    variant,
    ...rest
}) => {
    const baggageInfo = useBaggageInfo(variant);

    const baggageText = useMemo(() => {
        if (!baggageInfo.included) {
            return i18nBlock.withoutBaggage();
        }

        const weight = baggageInfo.wt?.count;
        const count = baggageInfo.pc?.count;

        if (!count) {
            return i18nBlock.withoutBaggage();
        }

        if (!weight) {
            return i18nBlock.withBaggage();
        }

        return `${i18nBlock.baggage()} ${count}${CHAR_MULTIPLICATION_SIGN}${weight} ${i18nUnitsBlock.kg()}`;
    }, [baggageInfo]);

    return (
        <Flex
            className={cx('root', {included: variant.hasBaggage}, className)}
            alignItems="center"
            between="2"
            inline
            nowrap
            textSize={textSize}
            {...prepareQaAttributes({parent: rest, current: 'baggageInfo'})}
        >
            <BaggageIconWithWeight
                type={EBaggageWithWeightIconType.BAGGAGE}
                included={variant.hasBaggage}
                {...prepareQaAttributes({parent: rest, current: 'baggageIcon'})}
            />

            <span
                {...prepareQaAttributes({parent: rest, current: 'baggageText'})}
            >
                {baggageText}
            </span>
        </Flex>
    );
};
