import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {formatSize} from 'projects/avia/lib/flightFareTerms/formatters';
import {CHAR_MULTIPLICATION_SIGN} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nUnitsBlock from 'i18n/common-units';
import * as i18nBlock from 'i18n/avia-baggage-factory';

import Flex from 'components/Flex/Flex';
import BaggageIconWithWeight, {
    EBaggageWithWeightIconType,
} from 'components/BaggageIconWithWeight/BaggageIconWithWeight';
import {TTextSize} from 'components/Text/Text';

import cx from './AviaCarryOnInfo.scss';

interface IAviaCarryOnInfoProps extends IWithClassName, IWithQaAttributes {
    places: number;
    included: boolean;
    size?: string;
    weight?: number;
    nowrap?: boolean;
    textSize?: TTextSize;
    limitedCarryOn?: boolean;
}

export const AviaCarryOnInfo: React.FC<IAviaCarryOnInfoProps> = props => {
    const {
        places,
        included,
        size,
        weight,
        nowrap = false,
        textSize = 'm',
        className,
        limitedCarryOn,
    } = props;

    return (
        <Flex
            className={cx('root', {included}, className)}
            between={2}
            inline
            nowrap
            {...prepareQaAttributes({parent: props, current: 'carryOnInfo'})}
        >
            <BaggageIconWithWeight
                type={EBaggageWithWeightIconType.CARRY_ON}
                included={included}
                weight={included ? undefined : weight}
                count={included ? undefined : places}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'carryOnIcon',
                })}
            />

            <Flex
                flexDirection={nowrap ? 'row' : 'column'}
                above="1"
                between="1"
                textSize={textSize}
            >
                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'carryOnText',
                    })}
                >
                    {i18nBlock.carryon()}{' '}
                    {Boolean(places && weight) &&
                        `${places}${CHAR_MULTIPLICATION_SIGN}${weight} ${i18nUnitsBlock.kg()}`}
                </span>
                {nowrap && <>&nbsp;</>}
                {limitedCarryOn && (
                    <span>{size ? formatSize(size) : <>&nbsp;</>}</span>
                )}
            </Flex>
        </Flex>
    );
};
