import React, {useMemo, useRef} from 'react';
import {useSelector} from 'react-redux';

import {isOnlyOnePassengerSelector} from 'selectors/avia/aviaPassengersCountSelector';

import {getAviaOfferType} from 'projects/avia/lib/search/offerType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {
    getVariantAviaCompanies,
    isSameCompanyForVariant,
} from 'projects/avia/lib/search/aviaCompanies';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useExperiments} from 'utilities/hooks/useExperiments';

import {
    allPassengers,
    learnMoreAboutTariffs,
    selectTariff,
} from 'i18n/avia-search';
import * as i18nBlock from 'i18n/avia-AviaResultVariant';

import Flex from 'components/Flex/Flex';
import CardWithBadges from 'projects/avia/components/AviaResultVariant/components/CardWithBadges/CardWithBadges';
import LogosWithTitle from 'projects/avia/components/AviaResultVariant/components/LogosWithTitle/LogosWithTitle';
import DesktopFlights from 'projects/avia/components/AviaResultVariant/components/DesktopFlights/DesktopFlights';
import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Tariffs from '../TariffSelector/components/Tariffs/Tariffs';
import Modal from 'components/Modal/Modal';
import OrderLink from 'projects/avia/components/OrderLink/OrderLink';

import {useOnBuyHandler} from 'projects/avia/hooks/useOnBuyHandler';

import DirectionInfo from '../DirectionInfo/DirectionInfo';
import {VariantPriceButton} from '../VariantPriceButton/VariantPriceButton';
import {AviaCarryOnInfo} from '../AviaCarryOnInfo/AviaCarryOnInfo';
import {AviaBaggageInfo} from '../AviaBaggageInfo/AviaBaggageInfo';
import useCarryOnInfo from '../../hooks/useCarryOnInfo';
import {IAviaResultVariantProps} from '../../AviaResultVariantContainer';

import cx from './AviaDesktopResultVariant.scss';

const AviaDesktopResultVariant: React.FC<IAviaResultVariantProps> = props => {
    const {
        group: {type, variants: groupVariants},
        selectedVariant,
        baggageToggleIsActive,
        variants,
        onVariantPriceButtonClick,
    } = props;

    const offerType = getAviaOfferType(groupVariants);

    const isOneWay = selectedVariant.route[1].length === 0;
    const showCommonCompany =
        !isOneWay && isSameCompanyForVariant(selectedVariant);

    const onlyOnePassenger = useSelector(isOnlyOnePassengerSelector);

    const buttonRef = useRef<HTMLElement>(null);

    const qa = useMemo(() => prepareQaAttributes(props), [props]);

    const {
        value: isTariffPopupVisible,
        toggle: toggleTariffPopup,
        setFalse: closeTariffPopup,
    } = useBoolean(false);

    const onBuyClick = useOnBuyHandler();

    const {
        aviaSearchButtonTextInside,
        aviaSearchButtonTextOutside,
        aviaSearchButtonTextOutsideSelect,
    } = useExperiments();
    const variantButtonText =
        onlyOnePassenger && aviaSearchButtonTextInside
            ? learnMoreAboutTariffs()
            : undefined;

    const variantLogo = useMemo(
        () =>
            showCommonCompany && (
                <LogosWithTitle
                    className={cx('companyLogo')}
                    alignItems="center"
                    companies={getVariantAviaCompanies(selectedVariant)}
                    {...prepareQaAttributes({
                        parent: qa,
                        current: 'airline',
                    })}
                />
            ),
        [selectedVariant, qa, showCommonCompany],
    );

    const [forwardRoute, backwardRoute] = selectedVariant.route;

    const route = useMemo(
        () =>
            selectedVariant.route.map(
                (flights, idx) =>
                    Boolean(flights.length) && (
                        <DesktopFlights
                            flights={flights}
                            variant={selectedVariant}
                            showAirlines={!showCommonCompany}
                            key={idx}
                            withBaggage={baggageToggleIsActive}
                            {...prepareQaAttributes({
                                parent: qa,
                                current:
                                    idx === 0
                                        ? 'forwardFlights'
                                        : 'backwardFlights',
                            })}
                        />
                    ),
            ),
        [baggageToggleIsActive, selectedVariant, qa, showCommonCompany],
    );

    const searchButtonTextOutsideText = useMemo(() => {
        if (aviaSearchButtonTextOutside) {
            return learnMoreAboutTariffs();
        }

        if (aviaSearchButtonTextOutsideSelect) {
            return selectTariff();
        }

        return null;
    }, [aviaSearchButtonTextOutside, aviaSearchButtonTextOutsideSelect]);

    const passengersHint = useMemo(
        () =>
            onlyOnePassenger ? (
                Boolean(searchButtonTextOutsideText) && (
                    <Text
                        className={cx('learnMoreAboutTariffs')}
                        size="s"
                        tag="div"
                        align="center"
                    >
                        {searchButtonTextOutsideText}
                    </Text>
                )
            ) : (
                <Text
                    className={cx('allPassengers')}
                    size="s"
                    tag="div"
                    align="center"
                >
                    {allPassengers()}
                </Text>
            ),
        [aviaSearchButtonTextOutside, onlyOnePassenger],
    );

    const carryOn = useCarryOnInfo(selectedVariant);

    return (
        <CardWithBadges
            className={cx('root')}
            type={type}
            offerType={offerType}
            price={selectedVariant.price}
            {...qa}
        >
            <Flex justifyContent="space-between">
                {variantLogo}

                <Flex
                    className={cx('route')}
                    alignItems="flex-start"
                    justifyContent="center"
                    flexDirection="column"
                    between="10"
                >
                    {route}
                </Flex>

                <Box
                    className={cx(
                        'selectedVariantInfo',
                        'selectedVariantInfoTariffExperiment',
                    )}
                    between="5"
                    onClick={toggleTariffPopup}
                >
                    <Box between={onlyOnePassenger ? 5 : 3}>
                        <Box>
                            <Box between="2">
                                <AviaCarryOnInfo {...carryOn} {...qa} />

                                <AviaBaggageInfo
                                    className={cx('baggage')}
                                    variant={selectedVariant}
                                    {...qa}
                                />

                                <VariantPriceButton
                                    className={cx('button')}
                                    price={selectedVariant.price}
                                    buttonRef={buttonRef}
                                    text={variantButtonText}
                                    onClick={onVariantPriceButtonClick}
                                    {...qa}
                                />

                                <Modal
                                    containerClassName={cx('modalContent')}
                                    isVisible={isTariffPopupVisible}
                                    onClose={closeTariffPopup}
                                    {...prepareQaAttributes({
                                        parent: qa,
                                        current: 'tariffSelectorPopup',
                                    })}
                                >
                                    <Flex between="8" flexDirection="column">
                                        <DirectionInfo
                                            direction={forwardRoute}
                                            variant={selectedVariant}
                                        />
                                        {Boolean(backwardRoute.length) && (
                                            <DirectionInfo
                                                direction={backwardRoute}
                                                variant={selectedVariant}
                                            />
                                        )}

                                        <Flex
                                            between="5"
                                            flexDirection="column"
                                        >
                                            <Text size="xl" weight="bold">
                                                {i18nBlock.tariffListHeader()}
                                            </Text>

                                            <Tariffs
                                                offers={groupVariants}
                                                variants={variants}
                                                onPriceButtonClick={onBuyClick}
                                                showPriceButton
                                                {...qa}
                                            />
                                        </Flex>
                                    </Flex>
                                </Modal>
                            </Box>

                            {passengersHint}

                            {selectedVariant.price.plusPoints && (
                                <Text size="s" className={cx('cashback')}>
                                    {i18nBlock.cashbackYandexPay()}
                                </Text>
                            )}
                        </Box>
                    </Box>
                </Box>
            </Flex>

            <OrderLink
                variant={selectedVariant}
                className={cx('orderLink')}
                withBaggage={selectedVariant.hasBaggage}
                {...prepareQaAttributes({parent: qa, current: 'link'})}
            />
        </CardWithBadges>
    );
};

export default AviaDesktopResultVariant;
