import React, {useCallback, useMemo, useRef} from 'react';
import {useSelector} from 'react-redux';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {isOnlyOnePassengerSelector} from 'selectors/avia/aviaPassengersCountSelector';

import {getQa, prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getAviaOfferType} from 'projects/avia/lib/search/offerType';
import {getVariantAviaCompanies} from 'projects/avia/lib/search/aviaCompanies';
import {CHAR_LIST_MARKER} from 'utilities/strings/charCodes';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useExperiments} from 'utilities/hooks/useExperiments';

import * as i18nBlock from 'i18n/avia-AviaResultVariant';
import {selectTariff} from 'i18n/avia-search';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Flights from 'projects/avia/components/AviaResultVariant/components/Flights/Flights';
import TariffSelectorBottomSheet from '../TariffSelector/components/TariffSelectorPopup/TariffSelectorBottomSheet';
import OrderLink from 'projects/avia/components/OrderLink/OrderLink';
import AirlineLogos from 'projects/avia/components/AirlineLogos/AirlineLogos';
import YandexPlusSignColorGradientS from 'icons/illustrations/YandexPlusSignColorGradient/YandexPlusSignColorGradientS';
import {VariantPriceButton} from 'projects/avia/components/AviaResultVariant/components/VariantPriceButton/VariantPriceButton';
import {useMarginMobileVariantBottom} from 'projects/avia/components/AviaResultVariant/components/AviaMobileResultVariant/hooks/useMarginMobileVariantBottom';

import {useOnBuyHandler} from 'projects/avia/hooks/useOnBuyHandler';

import CardWithBadges from '../CardWithBadges/CardWithBadges';
import ShortBaggageInfo from '../ShortBaggageInfo/ShortBaggageInfo';
import {IAviaResultVariantProps} from '../../AviaResultVariantContainer';

import cx from './AviaMobileResultVariant.scss';

const AviaMobileResultVariant: React.FC<IAviaResultVariantProps> = props => {
    const {
        group: {type, variants: groupVariants},
        selectedVariant,
        variants,
        onVariantPriceButtonClick,
    } = props;

    const {
        aviaTariffCardClickOpenOrderPage,
        aviaTariffCardClickOpenTariffSheet,
        aviaSearchButtonTextOutside,
    } = useExperiments();
    const linkRef = useRef<HTMLDivElement | null>(null);
    const buttonRef = useRef<HTMLDivElement | null>(null);

    const onBuyClick = useOnBuyHandler();

    const offerType = getAviaOfferType(groupVariants);

    const {
        value: showTariffSelector,
        setTrue: openTariffSelector,
        setFalse: closeTariffSelector,
    } = useBoolean(false);

    const variantPriceButtonClickHandler = useCallback(() => {
        openTariffSelector();
        onVariantPriceButtonClick?.();
    }, [openTariffSelector, onVariantPriceButtonClick]);

    const onlyOnePassenger = useSelector(isOnlyOnePassengerSelector);
    const isButtonSubtextVisible = Boolean(
        aviaSearchButtonTextOutside && onlyOnePassenger,
    );

    const marginBottom = useMarginMobileVariantBottom(
        buttonRef,
        linkRef,
        isButtonSubtextVisible,
    );

    const onTariffPriceButtonClick = useCallback(
        (variant: IResultAviaVariant) => {
            onBuyClick(variant);
            closeTariffSelector();
        },
        [onBuyClick, closeTariffSelector],
    );

    const qaParent = getQa(props);

    const flightsEl = useMemo(
        () => (
            <Flex flexDirection="column" between="1">
                {selectedVariant.route.map((flights, index) =>
                    flights.length ? (
                        <Flights
                            key={index}
                            flights={flights}
                            variant={selectedVariant}
                            {...prepareQaAttributes({
                                parent: qaParent,
                                current: 'flights',
                                key: index,
                            })}
                        />
                    ) : null,
                )}
            </Flex>
        ),
        [qaParent, selectedVariant],
    );

    const aviaCompanies = useMemo(
        () => getVariantAviaCompanies(selectedVariant),
        [selectedVariant],
    );

    const logos = useMemo(
        () =>
            aviaCompanies.map(aviaCompany => ({
                svg: aviaCompany.logoSvg,
                title: aviaCompany.title,
            })),
        [aviaCompanies],
    );

    const title = useMemo(
        () =>
            aviaCompanies
                .map(aviaCompany => aviaCompany.title)
                .join(` ${CHAR_LIST_MARKER} `),
        [aviaCompanies],
    );

    if (aviaTariffCardClickOpenOrderPage) {
        return (
            <OrderLink
                variant={selectedVariant}
                className={cx('orderLink')}
                {...prepareQaAttributes({
                    parent: qaParent,
                    current: 'link',
                })}
            >
                <CardWithBadges
                    shadow="none"
                    type={type}
                    offerType={offerType}
                    price={selectedVariant.price}
                    {...prepareQaAttributes(props)}
                >
                    <Flex flexDirection="column">
                        <Flex nowrap below="4">
                            <div className={cx('airlineInfo')}>
                                <Flex className={cx('logos')}>
                                    <AirlineLogos
                                        items={logos}
                                        size="s"
                                        {...prepareQaAttributes({
                                            parent: qaParent,
                                            current: 'logo',
                                        })}
                                    />

                                    <Text
                                        size="m"
                                        className={cx('title')}
                                        {...prepareQaAttributes({
                                            parent: qaParent,
                                            current: 'title',
                                        })}
                                    >
                                        {title}
                                    </Text>
                                </Flex>
                            </div>

                            <Flex onClick={openTariffSelector}>
                                <ShortBaggageInfo
                                    variant={selectedVariant}
                                    {...prepareQaAttributes({
                                        parent: qaParent,
                                        current: 'baggageInfo',
                                    })}
                                />
                            </Flex>
                        </Flex>

                        <Flex
                            nowrap
                            alignItems="center"
                            style={{
                                marginBottom: `${marginBottom}px`,
                            }}
                        >
                            <Box innerRef={linkRef} className={cx('flights')}>
                                {flightsEl}
                            </Box>

                            <Flex
                                innerRef={buttonRef}
                                flexDirection="column"
                                alignItems="center"
                            >
                                <VariantPriceButton
                                    className={cx('button')}
                                    price={selectedVariant.price}
                                    {...prepareQaAttributes(props)}
                                />
                                {isButtonSubtextVisible && (
                                    <Text
                                        size="s"
                                        tag="div"
                                        align="center"
                                        className={cx('selectTariff')}
                                    >
                                        {selectTariff()}
                                    </Text>
                                )}
                            </Flex>
                        </Flex>
                    </Flex>

                    {selectedVariant.price.plusPoints && (
                        <Text size="s" className={cx('cashbackInfo')}>
                            <YandexPlusSignColorGradientS
                                className={cx('cashbackIcon')}
                                width="12"
                                height="12"
                            />
                            {i18nBlock.cashbackYandexPay()}
                        </Text>
                    )}

                    <TariffSelectorBottomSheet
                        isOpen={showTariffSelector}
                        offers={groupVariants}
                        variants={variants}
                        onPriceButtonClick={onTariffPriceButtonClick}
                        onClose={closeTariffSelector}
                        selectedVariant={selectedVariant}
                        {...prepareQaAttributes({
                            parent: qaParent,
                            current: 'tariffSelectorPopup',
                        })}
                    />
                </CardWithBadges>
            </OrderLink>
        );
    }

    if (aviaTariffCardClickOpenTariffSheet) {
        return (
            <CardWithBadges
                shadow="none"
                type={type}
                offerType={offerType}
                price={selectedVariant.price}
                onClick={openTariffSelector}
                {...prepareQaAttributes(props)}
            >
                <Flex flexDirection="column">
                    <Flex nowrap below="4">
                        <div className={cx('airlineInfo')}>
                            <Flex className={cx('logos')}>
                                <AirlineLogos
                                    items={logos}
                                    size="s"
                                    {...prepareQaAttributes({
                                        parent: qaParent,
                                        current: 'logo',
                                    })}
                                />

                                <Text
                                    size="m"
                                    className={cx('title')}
                                    {...prepareQaAttributes({
                                        parent: qaParent,
                                        current: 'title',
                                    })}
                                >
                                    {title}
                                </Text>
                            </Flex>
                        </div>

                        <Flex onClick={openTariffSelector}>
                            <ShortBaggageInfo
                                variant={selectedVariant}
                                {...prepareQaAttributes({
                                    parent: qaParent,
                                    current: 'baggageInfo',
                                })}
                            />
                        </Flex>
                    </Flex>

                    <Flex
                        nowrap
                        alignItems="center"
                        style={{
                            marginBottom: `${marginBottom}px`,
                        }}
                    >
                        <Box innerRef={linkRef} className={cx('flights')}>
                            {flightsEl}
                        </Box>

                        <Flex
                            innerRef={buttonRef}
                            flexDirection="column"
                            alignItems="center"
                        >
                            <VariantPriceButton
                                className={cx('button')}
                                price={selectedVariant.price}
                                {...prepareQaAttributes(props)}
                            />
                            {isButtonSubtextVisible && (
                                <Text
                                    size="s"
                                    tag="div"
                                    align="center"
                                    className={cx('selectTariff')}
                                >
                                    {selectTariff()}
                                </Text>
                            )}
                        </Flex>
                    </Flex>
                </Flex>

                {selectedVariant.price.plusPoints && (
                    <Text size="s" className={cx('cashbackInfo')}>
                        <YandexPlusSignColorGradientS
                            className={cx('cashbackIcon')}
                            width="12"
                            height="12"
                        />
                        {i18nBlock.cashbackYandexPay()}
                    </Text>
                )}

                <TariffSelectorBottomSheet
                    isOpen={showTariffSelector}
                    offers={groupVariants}
                    variants={variants}
                    onPriceButtonClick={onTariffPriceButtonClick}
                    onClose={closeTariffSelector}
                    selectedVariant={selectedVariant}
                    {...prepareQaAttributes({
                        parent: qaParent,
                        current: 'tariffSelectorPopup',
                    })}
                />
            </CardWithBadges>
        );
    }

    return (
        <CardWithBadges
            shadow="none"
            type={type}
            offerType={offerType}
            price={selectedVariant.price}
            {...prepareQaAttributes(props)}
        >
            <Flex flexDirection="column">
                <Flex nowrap below="4">
                    <OrderLink
                        variant={selectedVariant}
                        className={cx('orderLink')}
                        {...prepareQaAttributes({
                            parent: qaParent,
                            current: 'link',
                        })}
                    >
                        <Flex className={cx('logos')}>
                            <AirlineLogos
                                items={logos}
                                size="s"
                                {...prepareQaAttributes({
                                    parent: qaParent,
                                    current: 'logo',
                                })}
                            />

                            <Text
                                size="m"
                                className={cx('title')}
                                {...prepareQaAttributes({
                                    parent: qaParent,
                                    current: 'title',
                                })}
                            >
                                {title}
                            </Text>
                        </Flex>
                    </OrderLink>

                    <Flex onClick={openTariffSelector}>
                        <ShortBaggageInfo
                            variant={selectedVariant}
                            {...prepareQaAttributes({
                                parent: qaParent,
                                current: 'baggageInfo',
                            })}
                        />
                    </Flex>
                </Flex>

                <Flex
                    nowrap
                    alignItems="center"
                    style={{
                        marginBottom: `${marginBottom}px`,
                    }}
                >
                    <OrderLink
                        variant={selectedVariant}
                        className={cx('orderLink')}
                        {...prepareQaAttributes({
                            parent: qaParent,
                            current: 'link',
                        })}
                    >
                        <Box innerRef={linkRef} className={cx('flights')}>
                            {flightsEl}
                        </Box>
                    </OrderLink>

                    <Flex
                        innerRef={buttonRef}
                        flexDirection="column"
                        alignItems="center"
                    >
                        <VariantPriceButton
                            className={cx('button')}
                            price={selectedVariant.price}
                            onClick={variantPriceButtonClickHandler}
                            {...prepareQaAttributes(props)}
                        />
                        {isButtonSubtextVisible && (
                            <Text
                                size="s"
                                tag="div"
                                align="center"
                                className={cx('selectTariff')}
                                onClick={openTariffSelector}
                            >
                                {selectTariff()}
                            </Text>
                        )}
                    </Flex>
                </Flex>
            </Flex>
            {selectedVariant.price.plusPoints && (
                <Text size="s" className={cx('cashbackInfo')}>
                    <YandexPlusSignColorGradientS
                        className={cx('cashbackIcon')}
                        width="12"
                        height="12"
                    />
                    {i18nBlock.cashbackYandexPay()}
                </Text>
            )}
            <TariffSelectorBottomSheet
                isOpen={showTariffSelector}
                offers={groupVariants}
                variants={variants}
                onPriceButtonClick={onTariffPriceButtonClick}
                onClose={closeTariffSelector}
                selectedVariant={selectedVariant}
                {...prepareQaAttributes({
                    parent: qaParent,
                    current: 'tariffSelectorPopup',
                })}
            />
        </CardWithBadges>
    );
};

export default React.memo(AviaMobileResultVariant);
