import React, {useMemo} from 'react';
import {lowerFirst} from 'lodash';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ISearchSegmentBadge} from 'components/SearchSegmentBadges/types/ISearchSegmentBadge';

import {EAviaVariantGroupType} from 'selectors/avia/utils/denormalization/variantGroup';
import {IResultVariantPrice} from 'selectors/avia/utils/denormalization/prices';
import {getMinPrice} from 'selectors/avia/search/minPriceSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {EAviaOfferType} from 'projects/avia/lib/search/offerType';

import * as i18nBlock from 'i18n/avia-flight';
import * as i18nAviaCharterHintBlock from 'i18n/avia-charter-hint';

import Card, {TCardShadow} from 'components/Card/Card';
import {ELabelThemeType} from 'components/Label/Label';
import SearchSegmentBadges from 'components/SearchSegmentBadges/SearchSegmentBadges';

import cx from './CardWithBadges.scss';

interface ICardWithBadgesProps extends IWithClassName, IWithQaAttributes {
    type: EAviaVariantGroupType | undefined;
    offerType?: EAviaOfferType;
    price: IResultVariantPrice;
    onClick?(e: React.MouseEvent<HTMLDivElement>): void;
    children: React.ReactNode;
    shadow?: TCardShadow;
}

const makeBadge = (
    text: string,
    theme: ELabelThemeType = ELabelThemeType.NORMAL,
): ISearchSegmentBadge =>
    ({
        size: 's',
        theme: theme,
        text: lowerFirst(text),
    } as const);

const CardWithBadges: React.FC<ICardWithBadgesProps> = props => {
    const {
        className,
        type,
        offerType,
        price,
        onClick,
        shadow = 'default',
        children,
    } = props;
    const minPrice = useSelector(getMinPrice);
    const {isMobile} = useDeviceType();

    const badges = useMemo(() => {
        const items: ISearchSegmentBadge[] = [];

        const isCheapestVariant = price.tariff.value === minPrice;
        const isBoy = price.boy;
        const isAviaCompanySeller = price.fromCompany && !isBoy;

        if (isCheapestVariant) {
            items.push(
                makeBadge(
                    i18nBlock[EAviaVariantGroupType.cheapest](),
                    ELabelThemeType.PRIMARY,
                ),
            );
        }

        if (isBoy) {
            items.push(makeBadge(i18nBlock[EAviaVariantGroupType.boy]()));
        }

        if (isAviaCompanySeller) {
            items.push(
                makeBadge(i18nBlock[EAviaVariantGroupType.aviacompany]()),
            );
        }

        if (
            type &&
            [
                EAviaVariantGroupType.comfy,
                EAviaVariantGroupType.popular,
            ].includes(type)
        ) {
            items.push(makeBadge(i18nBlock[type]()));
        }

        if (offerType) {
            const text =
                offerType === EAviaOfferType.charter
                    ? i18nAviaCharterHintBlock.charter()
                    : i18nAviaCharterHintBlock.specialDashConditions();

            items.push(makeBadge(text));
        }

        return items.slice(0, 2);
    }, [type, offerType, price, minPrice]);

    return (
        <Card
            className={cx(
                'root',
                {
                    root_withBadges: Boolean(badges.length),
                },
                className,
            )}
            shadow={shadow}
            x={isMobile ? 4 : 8}
            y={isMobile ? 4 : 7}
            onClick={onClick}
            {...prepareQaAttributes(props)}
        >
            <SearchSegmentBadges
                items={badges}
                {...prepareQaAttributes({parent: props, current: 'badges'})}
            />

            {children}
        </Card>
    );
};

export default React.memo(CardWithBadges);
