import React from 'react';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {parseDate} from 'utilities/dateUtils';
import {HUMAN_SHORT, TIME} from 'utilities/dateUtils/formats';
import {getDuration} from 'projects/avia/lib/date/getDuration';
import {getSegmentAviaCompanies} from 'projects/avia/lib/search/aviaCompanies';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/avia-Duration';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import TransferDataWithHint from 'projects/avia/components/AviaResultVariant/components/TransferDataWithHint/TransferDataWithHint';
import LogosWithTitle from 'projects/avia/components/AviaResultVariant/components/LogosWithTitle/LogosWithTitle';

import cx from './DesktopFlights.scss';

interface IDesktopFlightsProps extends IWithQaAttributes {
    flights: IResultAviaFlight[];
    variant: IResultAviaVariant;
    showAirlines: boolean;
    withBaggage?: boolean;
}

const DesktopFlights: React.FC<IDesktopFlightsProps> = props => {
    const {flights, variant, showAirlines, withBaggage} = props;
    const departure = flights[0].departure;
    const arrival = flights[flights.length - 1].arrival;

    if (flights.length === 0) {
        return null;
    }

    const firstFlight = flights[0];
    const lastFlight = flights[flights.length - 1];

    const parsedDepartureDate = parseDate(departure.local);
    const parsedArrivalDate = parseDate(arrival.local);
    const departureTime = parsedDepartureDate.utcOffset(departure.offset, true);
    const arrivalTime = parsedArrivalDate.utcOffset(arrival.offset, true);
    const dayIsChanged = departureTime.date() !== arrivalTime.date();
    const duration = getDuration(arrivalTime.diff(departureTime, 'ms'));
    const withTransfers = flights.length > 1;

    return (
        <Flex className={cx('root')} justifyContent="space-between">
            {showAirlines && (
                <LogosWithTitle
                    className={cx('head')}
                    companies={getSegmentAviaCompanies(flights)}
                    size="m"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'airline',
                    })}
                />
            )}

            <Box className={cx('body')} between="1">
                <Flex
                    className={cx('time')}
                    inline
                    alignItems="center"
                    justifyContent="space-between"
                    between="3"
                >
                    <Text
                        className={cx('departureTime')}
                        size="xl"
                        weight="bold"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'departureTime',
                        })}
                    >
                        {parsedDepartureDate.format(TIME)}
                    </Text>

                    <Flex
                        className={cx('durationWrapper')}
                        inline
                        alignItems="center"
                        between="2"
                    >
                        <Separator className={cx('separator')} />

                        <Text
                            className={cx('duration')}
                            color="secondary"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'duration',
                            })}
                        >
                            {i18nBlock.dDashHDashMDashFormat(duration).trim()}
                        </Text>

                        <Separator className={cx('separator')} />
                    </Flex>

                    <Flex
                        className={cx('arrivalTime')}
                        inline
                        alignItems="flex-start"
                        between="1"
                    >
                        <Text
                            size="xl"
                            weight="bold"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'arrivalTime',
                            })}
                        >
                            {parsedArrivalDate.format(TIME)}
                        </Text>

                        {dayIsChanged && (
                            <Text size="s" color="highlight">
                                {parsedArrivalDate.format(HUMAN_SHORT)}
                            </Text>
                        )}
                    </Flex>
                </Flex>

                <Flex
                    className={cx('points')}
                    inline
                    alignItems="flex-start"
                    justifyContent="space-between"
                >
                    <Box
                        className={cx('departurePoint', {
                            departurePoint_withTransfers: withTransfers,
                        })}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'departure',
                        })}
                    >
                        <Text className={cx('pointTitle')} tag="div">
                            {firstFlight.from.title}
                        </Text>
                        <Text tag="div">{firstFlight.from.code}</Text>
                    </Box>

                    <TransferDataWithHint
                        absoluteIcons
                        className={cx('transfers')}
                        directClassName={cx('direct')}
                        flights={flights}
                        variant={variant}
                        withBaggage={withBaggage}
                    />

                    <Box
                        className={cx('arrivalPoint')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'arrival',
                        })}
                    >
                        <Text className={cx('pointTitle')} tag="div">
                            {lastFlight.to.title}
                        </Text>
                        <Text tag="div">{lastFlight.to.code}</Text>
                    </Box>
                </Flex>
            </Box>
        </Flex>
    );
};

export default DesktopFlights;
