import moment from 'moment';
import React, {FC, useMemo} from 'react';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {getSegmentAviaCompanies} from 'projects/avia/lib/search/aviaCompanies';
import {CHAR_LIST_MARKER} from 'utilities/strings/charCodes';
import {HUMAN_DATE_WITH_FULL_WEEKDAY} from 'utilities/dateUtils/formats';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useExperiments} from 'utilities/hooks/useExperiments';

import Flex from 'components/Flex/Flex';
import AirlineLogos from 'projects/avia/components/AirlineLogos/AirlineLogos';
import Text from 'components/Text/Text';
import Raskrivashka from 'components/Raskrivashka/Raskrivashka';

import Flights from '../Flights/Flights';

import cx from './DirectionInfo.scss';

interface IDirectionInfoProps {
    direction: IResultAviaFlight[];
    variant: IResultAviaVariant;
    showStationCodeOnNewLine?: boolean;
}

const DirectionInfo: FC<IDirectionInfoProps> = ({
    direction,
    variant,
    showStationCodeOnNewLine,
}) => {
    const {aviaTariffFlightInfoAccordion} = useExperiments();

    const {departure} = direction[0];
    const {arrival} = direction[direction.length - 1];
    const fromTitle = direction[0].from.settlement?.title;
    const toTitle = direction[direction.length - 1].to.settlement?.title;

    const deviceType = useDeviceType();

    const companies = useMemo(
        () => getSegmentAviaCompanies(direction),
        [direction],
    );

    const airlineLogosItems = useMemo(
        () =>
            companies.map(aviaCompany => ({
                svg: aviaCompany.logoSvg,
                title: aviaCompany.title,
            })),
        [companies],
    );

    const airlineTitle = useMemo(
        () =>
            companies
                .map(aviaCompany => aviaCompany.title)
                .join(` ${CHAR_LIST_MARKER} `),
        [companies],
    );

    const header = useMemo(
        () => (
            <Text
                size={deviceType.isMobile ? 'l' : 'xl'}
                weight="bold"
                className={cx('title')}
                {...prepareQaAttributes({
                    current: 'points',
                })}
            >
                {fromTitle} — {toTitle}
            </Text>
        ),
        [fromTitle, toTitle, deviceType],
    );

    const date = useMemo(
        () => (
            <Text
                className={cx('date')}
                size="m"
                tag="div"
                {...prepareQaAttributes({current: 'dates'})}
            >
                {moment(departure.local).format(HUMAN_DATE_WITH_FULL_WEEKDAY)}
            </Text>
        ),
        [departure.local],
    );

    const dayIsChanged = useMemo(() => {
        const parsedDeparture = moment(departure.local).utcOffset(
            departure.offset,
            true,
        );
        const parsedArrival = moment(arrival.local).utcOffset(
            arrival.offset,
            true,
        );

        return !parsedArrival.isSame(parsedDeparture, 'day');
    }, [arrival.local, arrival.offset, departure.local, departure.offset]);

    const airlineInfo = useMemo(
        () => (
            <Flex
                className={cx({
                    airlineDayChangedFix: dayIsChanged,
                })}
                alignItems="center"
                between="2"
                below="2"
                inline
                nowrap
            >
                <AirlineLogos
                    items={airlineLogosItems}
                    size="m"
                    {...prepareQaAttributes({
                        current: 'icon',
                    })}
                />

                <Text
                    size="m"
                    className={cx('airlines')}
                    {...prepareQaAttributes({
                        current: 'title',
                    })}
                >
                    {airlineTitle}
                </Text>
            </Flex>
        ),
        [airlineLogosItems, airlineTitle, dayIsChanged],
    );

    const flights = useMemo(
        () => (
            <Flights
                className={cx('flights')}
                flights={direction}
                variant={variant}
                timeTextSize="l"
                stationTextSize="m"
                showStationTitle
                showStationCodeOnNewLine={showStationCodeOnNewLine}
                {...prepareQaAttributes({
                    current: 'flights',
                })}
            />
        ),
        [direction, showStationCodeOnNewLine, variant],
    );

    const containerClassName = cx(
        'directionInfo',
        deviceMods('directionInfo', deviceType),
    );

    if (deviceType.isDesktop)
        return (
            <Flex
                className={containerClassName}
                justifyContent="space-between"
                flexDirection="column"
                between="5"
                below="6"
            >
                <Flex inline alignItems="baseline" between="2">
                    {header}
                    {date}
                </Flex>

                <Flex justifyContent="space-between" alignItems="flex-start">
                    {airlineInfo}
                    {flights}
                </Flex>
            </Flex>
        );

    if (aviaTariffFlightInfoAccordion) {
        return (
            <Flex
                className={containerClassName}
                justifyContent="space-between"
                flexDirection="column"
                below="6"
            >
                <Raskrivashka
                    theme="clear"
                    label={
                        <>
                            {header}
                            {date}
                        </>
                    }
                    contentIsInvolved={false}
                    buttonClassName={cx('flightInfoAccordion_button')}
                    iconClassName={cx('flightInfoAccordion_icon')}
                >
                    <>
                        {airlineInfo}
                        {flights}
                    </>
                </Raskrivashka>
            </Flex>
        );
    }

    return (
        <Flex
            className={containerClassName}
            justifyContent="space-between"
            flexDirection="column"
            below="6"
        >
            {header}
            {date}
            {airlineInfo}
            {flights}
        </Flex>
    );
};

const DirectionInfoWrapper: FC<IDirectionInfoProps> = props => {
    if (!props.direction.length) return null;

    return React.createElement(DirectionInfo, props);
};

export default DirectionInfoWrapper;
