import React from 'react';
import moment from 'moment';

import {IWithClassName} from 'types/withClassName';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {HUMAN_SHORT, TIME} from 'utilities/dateUtils/formats';
import {getDuration} from 'projects/avia/lib/date/getDuration';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/avia-Duration';

import Text, {TTextSize} from 'components/Text/Text';
import DepartureAndArrivalInfo from 'components/DepartureAndArrivalInfo/DepartureAndArrivalInfo';
import TransfersCount from 'projects/avia/components/AviaResultVariant/components/Flights/components/TransfersCount/TransfersCount';
import TransfersInfo from 'projects/avia/components/AviaResultVariant/components/Flights/components/TransfersInfo/TransfersInfo';

import cx from './Flights.scss';

interface IFlightsProps extends IWithQaAttributes, IWithClassName {
    flights: IResultAviaFlight[];
    variant: IResultAviaVariant;
    timeTextSize?: TTextSize;
    stationTextSize?: TTextSize;
    showStationTitle?: boolean;
    showStationCodeOnNewLine?: boolean;
}

const Flights: React.FC<IFlightsProps> = props => {
    const {
        className,
        flights,
        variant,
        timeTextSize = 'm',
        stationTextSize = 's',
        showStationTitle = false,
        showStationCodeOnNewLine = false,
    } = props;

    if (!flights.length) return null;

    const {departure, from} = flights[0];
    const {arrival, to} = flights[flights.length - 1];

    const parsedDeparture = moment(departure.local).utcOffset(
        departure.offset,
        true,
    );
    const parsedArrival = moment(arrival.local).utcOffset(arrival.offset, true);

    const diffInMs = parsedArrival.diff(parsedDeparture);
    const dayIsChanged = parsedArrival.date() !== parsedDeparture.date();

    return (
        <DepartureAndArrivalInfo
            className={className}
            fromTime={
                <Text size={timeTextSize} weight="bold">
                    {parsedDeparture.format(TIME)}
                </Text>
            }
            fromTimeBottomDescription={
                <>
                    {showStationTitle && (
                        <Text size={stationTextSize} weight="normal">
                            {from.title}
                            {!showStationCodeOnNewLine && <>&nbsp;</>}
                        </Text>
                    )}
                    {showStationTitle && showStationCodeOnNewLine && <br />}
                    <Text size={stationTextSize} weight="normal">
                        {from.code}
                    </Text>
                </>
            }
            toTime={
                <Text size={timeTextSize} weight="bold">
                    {parsedArrival.format(TIME)}
                </Text>
            }
            toTimeBottomDescription={
                <>
                    {showStationTitle && (
                        <Text size={stationTextSize} weight="normal">
                            {to.title}
                            {!showStationCodeOnNewLine && <>&nbsp;</>}
                        </Text>
                    )}
                    {showStationTitle && showStationCodeOnNewLine && <br />}
                    <Text size={stationTextSize} weight="normal">
                        {to.code}
                    </Text>
                </>
            }
            toTimeTopDescription={
                dayIsChanged ? (
                    <Text size="s" color="highlight">
                        {parsedArrival.format(HUMAN_SHORT)}
                    </Text>
                ) : undefined
            }
            duration={
                <Text size="s">
                    {i18nBlock
                        .dDashHDashMDashFormat(getDuration(diffInMs, true))
                        .trim()}
                </Text>
            }
            durationBottomDescription={
                <TransfersCount
                    flights={flights}
                    className={cx('transfersCount')}
                />
            }
            footer={
                <TransfersInfo
                    flights={flights}
                    variant={variant}
                    className={cx('transfersInfo')}
                />
            }
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(Flights);
