import {partition} from 'lodash';
import {useSelector} from 'react-redux';
import {useCallback, useMemo} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {qidSelector} from 'selectors/avia/aviaSelectors';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {getAviaOfferType} from 'projects/avia/lib/search/offerType';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/avia-search-tariffs';

import BottomSheet from 'components/BottomSheet/BottomSheet';
import Text from 'components/Text/Text';
import AviaPriceList from 'projects/avia/components/AviaPriceList/AviaPriceList';
import Modal from 'components/Modal/Modal';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './OfferListPopup.scss';

interface IOfferListPopup {
    selectedVariant: IResultAviaVariant | null;
    offers: IResultAviaVariant[];
    show: boolean;
    onClose: () => void;
}

const OfferListPopup: React.FC<IOfferListPopup> = ({
    selectedVariant,
    offers,
    show,
    onClose,
}) => {
    const deviceType = useDeviceType();

    const qid = useSelector(qidSelector) ?? undefined;

    const coordinator = useCoordinator();

    const offerListData = useMemo(() => {
        const filteredOffers = offers.filter(Boolean);
        const [avia, other] = partition(
            filteredOffers.slice(1),
            variant => variant.price.fromCompany,
        );
        const offerType = getAviaOfferType(filteredOffers) ?? undefined;

        return {
            avia,
            other,
            offerType,
        };
    }, [offers]);

    const onOfferClick = useCallback(() => {
        reachGoal(EAviaGoal.TARIFF_OTHER_OFFER_CLICK);
        coordinator.doAction(EAppActions.REPORT_METRICS_EVENT, {
            name: EAviaGoal.TARIFF_OTHER_OFFER_CLICK,
        });
    }, [coordinator]);

    const content = useMemo(
        () =>
            selectedVariant ? (
                <>
                    <Text size="xl" weight="bold" className={cx('title')}>
                        {i18nBlock.allOffers()}
                    </Text>
                    <div className={cx('priceList')}>
                        <AviaPriceList
                            qid={qid}
                            offerType={offerListData.offerType}
                            cheapest={selectedVariant}
                            avia={offerListData.avia}
                            other={offerListData.other}
                            visibleCountOffers={Infinity}
                            onClick={onOfferClick}
                        />
                    </div>
                </>
            ) : (
                // чтобы шторка плавно закрывалась нужно оставить хоть какой то контент
                <></>
            ),
        [
            offerListData.avia,
            offerListData.offerType,
            offerListData.other,
            qid,
            selectedVariant,
            onOfferClick,
        ],
    );

    const popupClassName = cx(
        'offerListPopup',
        deviceMods('offerListPopup', deviceType),
    );

    if (deviceType.isMobile)
        return (
            <BottomSheet
                contentClassName={popupClassName}
                isOpened={show}
                onClose={onClose}
            >
                {content}
            </BottomSheet>
        );

    return (
        <Modal
            containerClassName={popupClassName}
            isVisible={show}
            onClose={onClose}
        >
            {content}
        </Modal>
    );
};

export default OfferListPopup;
