import React from 'react';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {formatSize} from 'projects/avia/lib/flightFareTerms/formatters';

import Flex from 'components/Flex/Flex';
import useCarryOnInfo from 'projects/avia/components/AviaResultVariant/hooks/useCarryOnInfo';
import useBaggageInfo from 'projects/avia/components/AviaResultVariant/hooks/useBaggageInfo';
import BaggageIconWithWeight, {
    EBaggageWithWeightIconType,
} from 'components/BaggageIconWithWeight/BaggageIconWithWeight';

import cx from './ShortBaggageInfo.scss';

interface IShortBaggageInfoProps extends IWithQaAttributes {
    variant: IResultAviaVariant;
    hideCarryOnSize?: boolean;
}

const ShortBaggageInfo: React.FC<IShortBaggageInfoProps> = props => {
    const {variant, hideCarryOnSize = false} = props;

    const carryOnInfo = useCarryOnInfo(variant);
    const baggageInfo = useBaggageInfo(variant);

    const hasBaggage = Boolean(baggageInfo.included?.count);
    const baggageWeight = (hasBaggage && baggageInfo.wt?.count) || undefined;
    const baggageCount = (hasBaggage && baggageInfo.pc?.count) || undefined;
    const showCarryOnSize =
        carryOnInfo.limitedCarryOn && carryOnInfo.size && !hideCarryOnSize;

    return (
        <Flex
            inline
            between={showCarryOnSize ? 0 : 2}
            nowrap
            {...prepareQaAttributes(props)}
        >
            <BaggageIconWithWeight
                type={EBaggageWithWeightIconType.CARRY_ON}
                included={carryOnInfo.included}
                weight={carryOnInfo.weight}
                count={carryOnInfo.places}
                currencyPrefix={CHAR_NBSP}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'carryOnIcon',
                })}
            />

            {showCarryOnSize && (
                <span className={cx('carryOnSize')}>
                    {formatSize(carryOnInfo.size)}
                </span>
            )}

            <BaggageIconWithWeight
                type={EBaggageWithWeightIconType.BAGGAGE}
                included={hasBaggage}
                weight={baggageWeight}
                count={baggageCount}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'baggageIcon',
                })}
            />
        </Flex>
    );
};

export default ShortBaggageInfo;
