import React, {FC, useCallback, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    reachChooseTariff,
    reachClickTariff,
    TTariffGoalContext,
} from 'projects/avia/utilities/tariffGoal';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import TariffSelectorPopup from 'projects/avia/components/AviaResultVariant/components/TariffSelector/components/TariffSelectorPopup/TariffSelectorPopup';
import Switcher from 'projects/avia/components/AviaResultVariant/components/TariffSelector/components/Switcher/Switcher';

interface ITariffSelectorProps extends IWithClassName, IWithQaAttributes {
    metricsContext: TTariffGoalContext;
    scope?: HTMLElement;
    variants: IResultAviaVariant[];
    selectedVariant: IResultAviaVariant;
    needToHideIfNoChoice?: boolean;
    onSelect(variant: IResultAviaVariant): void;
    plusPoints?: number;
}

const TariffSelector: FC<ITariffSelectorProps> = props => {
    const {
        className,
        variants,
        selectedVariant,
        needToHideIfNoChoice = false,
        onSelect,
        metricsContext,
        scope,
        plusPoints,
    } = props;
    const propsQa = getQa(props);

    const anchorRef = useRef<HTMLElement>(null);

    const {value: isOpen, setFalse: close, toggle} = useBoolean(false);

    const handleSelect = useCallback(
        (variantToSelect: IResultAviaVariant) => {
            close();

            reachChooseTariff(variantToSelect, metricsContext);

            onSelect(variantToSelect);
        },
        [metricsContext, close, onSelect],
    );

    const handleToggle = useCallback(() => {
        reachClickTariff(variants, metricsContext);

        toggle();
    }, [metricsContext, toggle, variants]);

    if (needToHideIfNoChoice && variants.length === 1) {
        return null;
    }

    return (
        <>
            <Switcher
                className={className}
                variantsCount={variants.length}
                selectedVariant={selectedVariant}
                plusPoints={plusPoints}
                anchorRef={anchorRef}
                onToggle={handleToggle}
                {...prepareQaAttributes({
                    parent: propsQa,
                    current: 'tariffSelectorButton',
                })}
            />

            <TariffSelectorPopup
                isOpen={isOpen}
                selectedVariant={selectedVariant}
                variants={variants}
                anchorRef={anchorRef}
                scope={scope}
                onSelect={handleSelect}
                onClose={close}
                {...prepareQaAttributes({
                    parent: propsQa,
                    current: 'tariffSelectorPopup',
                })}
            />
        </>
    );
};

export default React.memo(TariffSelector);
