import React, {FC, useMemo, RefObject} from 'react';

import {IWithClassName} from 'types/withClassName';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/avia-search-tariffs';

import SelectArrow from 'icons/16/SelectArrow';
import Button from 'components/Button/Button';
import LinkButton from 'components/LinkButton/LinkButton';
import Flex from 'components/Flex/Flex';
import TariffInfo from 'projects/avia/components/AviaResultVariant/components/TariffSelector/components/TariffInfo/TariffInfo';
import Box from 'components/Box/Box';
import BaggagePlusInfo from 'projects/avia/components/AviaResultVariant/components/AviaMobileResultVariant/components/BaggagePlusInfo/BaggagePlusInfo';

import cx from './Switcher.scss';

interface ISwitcherProps extends IWithClassName, IWithQaAttributes {
    selectedVariant: IResultAviaVariant;
    variantsCount: number;
    disabled?: boolean;
    plusPoints?: number;
    anchorRef: RefObject<HTMLElement>;
    onToggle(): void;
}

const Switcher: FC<ISwitcherProps> = ({
    className,
    selectedVariant,
    variantsCount,
    disabled = false,
    plusPoints,
    anchorRef,
    onToggle,
    ...rest
}) => {
    const deviceType = useDeviceType();

    const desktopContentBlock = useMemo(
        () => (
            <>
                <TariffInfo
                    selectedVariant={selectedVariant}
                    {...prepareQaAttributes(rest)}
                />

                {!disabled && <SelectArrow className={cx('selectArrowIcon')} />}
            </>
        ),
        [disabled, rest, selectedVariant],
    );

    if (deviceType.isMobile) {
        return (
            <Box
                className={cx('root', 'root_mobile', className)}
                between={2}
                onClick={disabled ? undefined : onToggle}
            >
                <Flex alignItems="center">
                    <LinkButton
                        className={cx('chooseTariff')}
                        disabled={disabled}
                    >
                        {variantsCount === 1
                            ? i18nBlock.moreInfo()
                            : i18nBlock.chooseTariff()}
                    </LinkButton>

                    <Flex className={cx('tariffInfo')} alignItems="center">
                        <TariffInfo selectedVariant={selectedVariant} />
                    </Flex>
                </Flex>

                {plusPoints && (
                    <BaggagePlusInfo
                        className={cx('plusPoints')}
                        plusPoints={plusPoints}
                    />
                )}
            </Box>
        );
    }

    if (disabled) {
        return (
            <div
                className={cx(
                    'root',
                    'root_desktop',
                    'root_disabled',
                    className,
                )}
            >
                {desktopContentBlock}
            </div>
        );
    }

    return (
        <Button
            className={cx('root', 'root_desktop', 'root_enabled', className)}
            theme="outlined"
            width="max"
            size="l"
            innerRef={anchorRef}
            onClick={onToggle}
            {...prepareQaAttributes(rest)}
        >
            {desktopContentBlock}
        </Button>
    );
};

export default React.memo(Switcher);
