import React, {FC, RefObject, useCallback} from 'react';
import {useSelector} from 'react-redux';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IAviaCarryOn} from 'types/avia/common/IAviaCarryOn';
import {IFFChangingCarriageTermRule} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {aviaAeroflotPlusSelector} from 'projects/avia/selectors/aviaAeroflotPlusEnabledSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';

import * as i18nPlusBlock from 'i18n/avia-plusPromo2021';
import * as i18nBlock from 'i18n/avia-search-tariffs';

import AviaVariantPrice from 'projects/avia/components/AviaVariantPrice/AviaVariantPrice';
import PlusPoints from 'components/PlusPoints/PlusPoints';
import Box from 'components/Box/Box';
import Card from 'components/Card/Card';
import Text from 'components/Text/Text';
import ExpandedRefundInfo from 'projects/avia/components/RefundInfo/ExpandedRefundInfo';
import {VariantPriceButton} from 'projects/avia/components/AviaResultVariant/components/VariantPriceButton/VariantPriceButton';
import LinkButton from 'components/LinkButton/LinkButton';
import Flex from 'components/Flex/Flex';
import ChangingCarriageInfo from 'projects/avia/components/ChangingCarriageInfo/ChangingCarriageInfo';

import {useCoordinator} from 'contexts/PlatformContext';

import {AviaBaggageInfo} from '../../../AviaBaggageInfo/AviaBaggageInfo';
import {AviaCarryOnInfo} from '../../../AviaCarryOnInfo/AviaCarryOnInfo';

import cx from './TariffItem.scss';

interface ITariffItemProps extends IWithQaAttributes {
    innerRef?: RefObject<HTMLElement>;
    variant: IResultAviaVariant;
    carryOn: IAviaCarryOn;
    changingCarriage?: IFFChangingCarriageTermRule;
    alwaysShowCarryOnSize?: boolean;
    isActive?: boolean;
    withTariffName: boolean;
    showPriceHeader: boolean;
    showPriceButton: boolean;
    footer?: React.ReactNode;
    countOffers?: number;
    highlightOnHover?: boolean;
    onClick?(variant: IResultAviaVariant): void;
    onMoreOffersClick?(variant: IResultAviaVariant): void;
    onPriceButtonClick?(variant: IResultAviaVariant): void;
}

const TariffItem: FC<ITariffItemProps> = props => {
    const {
        innerRef,
        variant,
        carryOn,
        changingCarriage,
        alwaysShowCarryOnSize = false,
        isActive,
        variant: {price},
        withTariffName,
        showPriceHeader,
        showPriceButton,
        countOffers = 0,
        highlightOnHover = false,
        onClick,
        onMoreOffersClick,
        onPriceButtonClick,
    } = props;

    const isAeroflotPlus = useSelector(aviaAeroflotPlusSelector);

    const coordinator = useCoordinator();

    const withPlus = Boolean(price.plusPoints);

    const handleClick = useCallback(
        () => onClick?.(variant),
        [onClick, variant],
    );

    const handlePriceButtonClick = useCallback(() => {
        onPriceButtonClick?.(variant);
        coordinator.doAction(EAppActions.REPORT_METRICS_EVENT, {
            name: EAviaGoal.ORDER_TARIFF_SELECTOR_BUTTON_CLICK,
        });
    }, [coordinator, onPriceButtonClick, variant]);

    const moreOffersClickHandler = useCallback(() => {
        onMoreOffersClick?.(variant);
        reachGoal(EAviaGoal.TARIFF_SHOW_OTHER_OFFERS_CLICK);
        coordinator.doAction(EAppActions.REPORT_METRICS_EVENT, {
            name: EAviaGoal.TARIFF_SHOW_OTHER_OFFERS_CLICK,
        });
    }, [coordinator, onMoreOffersClick, variant]);

    const carryOnInfoProps: IAviaCarryOn = alwaysShowCarryOnSize
        ? {...carryOn, limitedCarryOn: true}
        : carryOn;

    return (
        <Card
            className={cx('root', {
                root_withPlus: isAeroflotPlus && withPlus,
                root_active: isActive,
                highlightOnHover,
            })}
            ref={innerRef}
            onClick={handleClick}
            {...prepareQaAttributes(props)}
        >
            <Flex className={cx('wrapper')} flexDirection="column">
                {withTariffName && isAeroflotPlus && (
                    <Text
                        className={cx('tariffName')}
                        tag="div"
                        color={withPlus ? 'plus' : undefined}
                    >
                        {price.tariffName}
                    </Text>
                )}

                {showPriceHeader && (
                    <>
                        <AviaVariantPrice
                            className={cx('price')}
                            nationalPriceClassName={cx('nationalPrice')}
                            originPriceClassName={cx('originPrice')}
                            price={price}
                            theme="search"
                            flexDirection="row"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'price',
                            })}
                        />

                        {price.plusPoints && isAeroflotPlus ? (
                            <div className={cx('plusPoints')}>
                                <PlusPoints count={price.plusPoints} />{' '}
                                {i18nPlusBlock.pointForOrder()}
                            </div>
                        ) : null}
                    </>
                )}

                <Box between="2" below="3">
                    <AviaCarryOnInfo
                        {...carryOnInfoProps}
                        {...prepareQaAttributes(props)}
                    />

                    <AviaBaggageInfo
                        variant={variant}
                        {...prepareQaAttributes(props)}
                    />

                    <ExpandedRefundInfo
                        className={cx('refund')}
                        refund={price.worstRefund}
                        {...prepareQaAttributes(props)}
                    />

                    {changingCarriage && (
                        <ChangingCarriageInfo
                            changingCarriage={changingCarriage}
                            {...prepareQaAttributes(props)}
                        />
                    )}
                </Box>

                {showPriceButton && (
                    <VariantPriceButton
                        className={cx('button')}
                        price={price}
                        showPartner
                        onClick={handlePriceButtonClick}
                        {...prepareQaAttributes(props)}
                    />
                )}

                {countOffers > 0 && (
                    <LinkButton
                        className={cx('showMoreOffersLink')}
                        onClick={moreOffersClickHandler}
                    >
                        {i18nBlock.moreOffers({
                            count: countOffers,
                        })}
                    </LinkButton>
                )}
            </Flex>
        </Card>
    );
};

export default React.memo(TariffItem);
