import React, {FC, useMemo} from 'react';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {useReachGoalThrottle} from 'utilities/metrika/useReachGoal';

import * as i18nBlock from 'i18n/avia-search-tariffs';

import BottomSheet from 'components/BottomSheet/BottomSheet';
import Text from 'components/Text/Text';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import Tariffs from 'projects/avia/components/AviaResultVariant/components/TariffSelector/components/Tariffs/Tariffs';
import SpaceSeparator from 'components/SpaceSeparator/SpaceSeparator';

import DirectionInfo from '../../../DirectionInfo/DirectionInfo';

import cx from './TariffSelectorPopup.scss';

interface ITariffSelectorBottomSheetProps extends IWithQaAttributes {
    isOpen: boolean;
    offers?: IResultAviaVariant[];
    variants: IResultAviaVariant[];
    selectedVariant: IResultAviaVariant;
    onPriceButtonClick?(variant: IResultAviaVariant): void;
    onClose(): void;
}

const TariffSelectorBottomSheet: FC<ITariffSelectorBottomSheetProps> =
    props => {
        const {
            isOpen,
            offers,
            variants,
            selectedVariant,
            onPriceButtonClick,
            onClose,
        } = props;

        const {aviaTariffHideFlight, aviaTariffMoveCardTop} = useExperiments();

        const [forwardRoute, backwardRoute] = selectedVariant.route;

        const directions = useMemo(() => {
            if (aviaTariffHideFlight) return null;

            return (
                <>
                    <DirectionInfo
                        direction={forwardRoute}
                        variant={selectedVariant}
                        showStationCodeOnNewLine
                    />
                    {Boolean(backwardRoute.length) && (
                        <DirectionInfo
                            direction={backwardRoute}
                            variant={selectedVariant}
                            showStationCodeOnNewLine
                        />
                    )}
                </>
            );
        }, [
            aviaTariffHideFlight,
            backwardRoute,
            forwardRoute,
            selectedVariant,
        ]);

        const onTariffScroll = useReachGoalThrottle(EAviaGoal.TARIFF_SCROLL);

        return (
            <BottomSheet
                isOpened={isOpen}
                onClose={onClose}
                contentClassName={cx('bottomSheetContainer')}
            >
                {!aviaTariffMoveCardTop && directions}

                <Text size="l" weight="bold">
                    {variants.length === 1
                        ? i18nBlock.aboutTariff()
                        : i18nBlock.chooseTariff()}
                </Text>

                <HorizontalScroller
                    className={cx('scroller')}
                    padderClassName={cx('scrollPadder')}
                    offset={4}
                    // Не передаем событие touchmove выше, потому что иначе возникают лишние
                    // движения шторки вверх-вниз, и часто при скролле тарифов по горизонтали
                    // шторка просто закрывается
                    stopPropagationTouchmove
                    onScroll={onTariffScroll}
                    {...prepareQaAttributes(props)}
                >
                    <Tariffs
                        offers={offers}
                        variants={variants}
                        onPriceButtonClick={onPriceButtonClick}
                        showPriceButton
                        {...prepareQaAttributes(props)}
                    />
                </HorizontalScroller>

                {aviaTariffMoveCardTop && (
                    <>
                        <SpaceSeparator space={4} />
                        {directions}
                    </>
                )}
            </BottomSheet>
        );
    };

const TariffSelectorBottomSheetWrapper: FC<ITariffSelectorBottomSheetProps> =
    props => {
        if (!props.selectedVariant) return null;

        return <TariffSelectorBottomSheet {...props} />;
    };

export default React.memo(TariffSelectorBottomSheetWrapper);
