import React, {FC, RefObject, useCallback, useMemo, useState} from 'react';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {EFareFamilyAvailability} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IResultVariantPrice} from 'selectors/avia/utils/denormalization/prices';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {reachGoal} from 'utilities/metrika';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import Carousel from 'components/Carousel/Carousel';
import {usePreparedVariants} from 'projects/avia/components/AviaResultVariant/components/TariffSelector/hooks/usePreparedVariants';

import TariffItem from '../TariffItem/TariffItem';
import OfferListPopup from '../../../OfferListPopup/OfferListPopup';

const isSamePriceOptions = (
    price1: IResultVariantPrice,
    price2: IResultVariantPrice,
): boolean =>
    price1.hasBaggage === price2.hasBaggage &&
    price1.hasFreeRefund === price2.hasFreeRefund &&
    price1.hasChargedRefund === price2.hasChargedRefund &&
    price1.hasNoRefund === price2.hasNoRefund;

interface ITariffsProps extends IWithQaAttributes {
    selectedVariant?: IResultAviaVariant | null;
    offers?: IResultAviaVariant[];
    variants: IResultAviaVariant[];
    activeItemRef?: RefObject<HTMLElement>;
    showPriceHeader?: boolean;
    showPriceButton?: boolean;
    highlightOnHover?: boolean;
    onSelect?(variant: IResultAviaVariant): void;
    onPriceButtonClick?(variant: IResultAviaVariant): void;
}

const Tariffs: FC<ITariffsProps> = props => {
    const {
        selectedVariant,
        offers,
        variants,
        activeItemRef,
        showPriceHeader = false,
        showPriceButton = false,
        highlightOnHover = false,
        onSelect,
        onPriceButtonClick,
    } = props;

    const {aviaTariffChangingCarriageEnabled} = useExperiments();

    const {isDesktop} = useDeviceType();

    const needToShowTariffName = useMemo(() => {
        let someTariffHasPlusPoints = false;

        const areAllTariffsWithName = variants.every(uniqVariant => {
            if (uniqVariant.price.plusPoints) {
                someTariffHasPlusPoints = true;
            }

            return uniqVariant.price.tariffName;
        });

        return someTariffHasPlusPoints && areAllTariffsWithName;
    }, [variants]);

    const [variantToShowOffers, setVariantToShowOffers] =
        useState<IResultAviaVariant | null>(null);

    const hideOffersSheet = useCallback(() => setVariantToShowOffers(null), []);

    const preparedTariffs = usePreparedVariants(variants);

    const alwaysShowCarryOnSize = useMemo(
        () => preparedTariffs.some(({carryOn}) => carryOn.limitedCarryOn),
        [preparedTariffs],
    );

    const selectedVariantOffers = useMemo(
        () =>
            offers && variantToShowOffers
                ? offers
                      .filter(offer =>
                          isSamePriceOptions(
                              variantToShowOffers.price,
                              offer.price,
                          ),
                      )
                      .sort((variant1, variant2) => {
                          const {
                              price: {
                                  boy: isBoy1,
                                  fromCompany: isFromCompany1 = false,
                                  tariff: {value: price1},
                              },
                          } = variant1;
                          const {
                              price: {
                                  boy: isBoy2,
                                  fromCompany: isFromCompany2 = false,
                                  tariff: {value: price2},
                              },
                          } = variant2;

                          if (price1 === price2) {
                              const boySort = Number(isBoy2) - Number(isBoy1);

                              if (boySort !== 0) return boySort;

                              const fromCompanySort =
                                  Number(isFromCompany2) -
                                  Number(isFromCompany1);

                              if (fromCompanySort !== 0) return fromCompanySort;

                              return 0;
                          }

                          return price1 - price2;
                      })
                : [],
        [offers, variantToShowOffers],
    );

    const initialVisibleIndex = useMemo(
        () =>
            isDesktop
                ? undefined
                : variants.findIndex(variant => variant === selectedVariant),
        [isDesktop, selectedVariant, variants],
    );

    const onPriceButtonClickHandler = useCallback(
        (variant: IResultAviaVariant) => {
            onPriceButtonClick?.(variant);

            const hasRefund =
                variant.price.worstRefund?.availability &&
                [
                    EFareFamilyAvailability.free,
                    EFareFamilyAvailability.charge,
                ].includes(variant.price.worstRefund?.availability);

            reachGoal(EAviaGoal.TARIFF_CLICK, {
                hasBaggage: variant.hasBaggage,
                hasRefund,
            });
        },
        [onPriceButtonClick],
    );

    const onTariffsCarouselScroll = useReachGoal(EAviaGoal.TARIFF_SCROLL);

    return (
        <>
            <Carousel
                initialVisibleIndex={initialVisibleIndex}
                spaceBetween={2}
                onScrollContent={onTariffsCarouselScroll}
            >
                {preparedTariffs.map(({variant, carryOn}, index) => {
                    const tariffOffers = offers
                        ? offers.filter(offer =>
                              isSamePriceOptions(variant.price, offer.price),
                          )
                        : [];
                    const count = tariffOffers.length;

                    const tariffChangingCarriage =
                        aviaTariffChangingCarriageEnabled
                            ? variant.price.changingCarriage
                            : undefined;

                    const isSelectedVariant = variant === selectedVariant;

                    return (
                        <TariffItem
                            key={index}
                            variant={variant}
                            carryOn={carryOn}
                            changingCarriage={tariffChangingCarriage}
                            isActive={isSelectedVariant}
                            alwaysShowCarryOnSize={alwaysShowCarryOnSize}
                            innerRef={
                                isSelectedVariant ? activeItemRef : undefined
                            }
                            showPriceButton={showPriceButton}
                            showPriceHeader={showPriceHeader}
                            withTariffName={needToShowTariffName}
                            onClick={onSelect}
                            onPriceButtonClick={onPriceButtonClickHandler}
                            countOffers={count > 0 ? count - 1 : undefined}
                            onMoreOffersClick={setVariantToShowOffers}
                            highlightOnHover={highlightOnHover}
                            {...prepareQaAttributes({
                                parent: props,
                                key: index,
                            })}
                        />
                    );
                })}
            </Carousel>

            <OfferListPopup
                selectedVariant={variantToShowOffers}
                offers={selectedVariantOffers}
                show={Boolean(variantToShowOffers)}
                onClose={hideOffersSheet}
            />
        </>
    );
};

export default React.memo(Tariffs);
