import React from 'react';
import moment from 'moment';

import {IWithClassName} from 'src/types/withClassName';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {getTime} from 'projects/avia/lib/date/getTime';
import {getDuration} from 'projects/avia/lib/date/getDuration';
import {getTypeOfAviaSegment} from 'projects/avia/lib/search/segments';
import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import {dDashHDashMDashFormat} from 'i18n/avia-Duration';
import * as i18nAviaSearchBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import ConnectionIcon from 'icons/12/Connection';
import NightConnectionIcon from 'icons/12/NightConnection';
import SelfConnectMessage from 'projects/avia/components/AviaResultVariant/components/SelfConnectMessage/SelfConnectMessage';
import SelfConnectDescription from 'projects/avia/components/AviaResultVariant/components/SelfConnectDescription/SelfConnectDescription';
import Text, {TTextColor, TTextSize} from 'components/Text/Text';
import {AviaCustomTransportType} from 'projects/avia/components/AviaCustomTransportType/AviaCustomTransportType';

import cx from './TransferData.scss';

interface ITransferDataProps extends IWithClassName, IWithQaAttributes {
    flights: IResultAviaFlight[];
    variant: IResultAviaVariant;
    size?: TTextSize;
    theme?: 'light' | 'dark';
    short?: boolean;
    absoluteIcons?: boolean;
    directClassName?: string;
}

interface IThemeColors {
    primary: TTextColor;
    secondary: TTextColor;
    attention: TTextColor;
}

const TransferData: React.FC<ITransferDataProps> = ({
    className,
    flights,
    variant,
    size = 'm',
    short = true,
    theme = 'light',
    absoluteIcons = false,
    directClassName,
}) => {
    if (flights.length === 0) {
        return null;
    }

    const firstFlight = flights[0];

    const isDarkTheme = theme === 'dark';
    const colors: IThemeColors = {
        primary: isDarkTheme ? 'inverse' : 'primary',
        secondary: isDarkTheme ? 'inverse' : 'secondary',
        attention: isDarkTheme ? 'inverse' : 'highlight',
    };

    return (
        <Flex
            className={cx(absoluteIcons && 'root_absoluteIcons', className)}
            flexDirection="column"
            between={short ? 0 : 3}
        >
            {flights.length > 1 ? (
                flights.map((flight, index) =>
                    flights[index - 1]
                        ? renderTransferData(flight, flights[index - 1], {
                              size,
                              colors,
                              short,
                              absoluteIcons,
                          })
                        : null,
                )
            ) : (
                <Text className={cx('row')} color={colors.primary} size={size}>
                    <AviaCustomTransportType
                        className={cx('icon')}
                        type={getTypeOfAviaSegment(
                            firstFlight.from.tType,
                            firstFlight.to.tType,
                        )}
                    />

                    <Text
                        className={directClassName}
                        color={colors.secondary}
                        size={size}
                    >
                        {` ${i18nAviaSearchBlock.transferDotDirect()}`}
                    </Text>
                </Text>
            )}
            {flights.length > 1 &&
                (short ? (
                    <SelfConnectMessage
                        className={cx('selfConnect')}
                        variant={variant}
                    />
                ) : (
                    <SelfConnectDescription variant={variant} />
                ))}
        </Flex>
    );
};

interface ITransferRenderProps {
    size: TTextSize;
    colors: IThemeColors;
    short: boolean;
    absoluteIcons: boolean;
}

function renderTransferData(
    flight: IResultAviaFlight,
    previousFlight: IResultAviaFlight,
    renderProps: ITransferRenderProps,
): React.ReactElement {
    const {size, colors, short} = renderProps;
    const {arrival, to} = previousFlight;
    const {departure, from} = flight;

    const airportChange = to.id !== from.id;
    const isNightTransfer = !moment(arrival.local).isSame(
        departure.local,
        'day',
    );

    const icons = [];

    if (isNightTransfer) {
        icons.push(
            <NightConnectionIcon
                key="night"
                className={cx('icon', 'icon_night')}
            />,
        );
    }

    if (flight.to.tType !== 'plane') {
        icons.push(
            <AviaCustomTransportType
                className={cx('icon')}
                key="transportType"
                type={flight.to.tType}
                withTitle={false}
            />,
        );
    }

    const duration = (
        <Text className={cx('duration')} color={colors.attention} size={size}>
            {`${dDashHDashMDashFormat(
                getDuration(getTime(departure) - getTime(arrival)),
            ).trim()}`}
        </Text>
    );

    return (
        <Box between={short ? 0 : 4} key={previousFlight.key}>
            <Text
                className={cx('row')}
                tag="div"
                color={colors.primary}
                size={size}
            >
                <span className={cx('icons')}>{icons}</span>
                <Text color={colors.primary} size={size}>
                    {short ? (
                        <>
                            {`${to.code} `}
                            {duration}
                        </>
                    ) : (
                        insertJSXIntoKey(i18nAviaSearchBlock.transferDotInfo)({
                            night: isNightTransfer,
                            point: to.settlement?.phraseIn ?? '',
                            isBus: flight.to.tType === 'bus',
                            isTrain: flight.to.tType === 'train',
                            duration,
                            preposition: to.settlement?.preposition ?? '',
                        })
                    )}
                </Text>
            </Text>

            {airportChange && (
                <Text
                    className={cx('row')}
                    tag="div"
                    color={colors.attention}
                    size={size}
                >
                    <span className={cx('icons')}>
                        <ConnectionIcon className={cx('icon')} />
                    </span>
                    <Text color={colors.primary} size={size}>
                        {` ${i18nAviaSearchBlock.transferDotAirportDashChange()}`}
                    </Text>
                </Text>
            )}
        </Box>
    );
}

export default TransferData;
