import React, {RefObject, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IResultVariantPrice} from 'selectors/avia/utils/denormalization/prices';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useCurrencyConverter} from 'utilities/hooks/useCurrencyConverter';
import {formatPrice} from 'utilities/currency';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlockPrice from 'i18n/common-price';
import * as i18nBlockVariant from 'i18n/avia-AviaResultVariant';

import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import Text, {TTextSize} from 'components/Text/Text';
import AviaVariantPrice from 'projects/avia/components/AviaVariantPrice/AviaVariantPrice';
import RedirectLink from 'projects/avia/components/RedirectLink/RedirectLink';
import PlusColoredLabel from 'components/PlusColoredLabel/PlusColoredLabel';
import BorderPositioned from 'components/BorderPositioned/BorderPositioned';

import cx from './VariantPriceButton.scss';

interface IPriceButtonProps extends IWithQaAttributes, IWithClassName {
    price: IResultVariantPrice;
    text?: string;
    fontSize?: TTextSize;
    buttonRef?: RefObject<HTMLElement>;
    onClick?: React.MouseEventHandler<HTMLElement>;
}

export const PriceButton: React.FC<IPriceButtonProps> = props => {
    const {price, text, fontSize, className, buttonRef, onClick} = props;

    const deviceType = useDeviceType();

    return (
        <div
            className={cx('priceButton', deviceMods('priceButton', deviceType))}
        >
            <Button
                className={cx('button', className)}
                theme="primary"
                size="l"
                width="max"
                innerRef={buttonRef}
                onClick={onClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'buyButton',
                })}
            >
                <Flex
                    flexDirection="column"
                    alignItems="center"
                    textColor="primary"
                >
                    <AviaVariantPrice
                        price={price}
                        theme="search"
                        flexDirection="row"
                        className={cx('aviaVariantPrice')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'price',
                        })}
                    />

                    <Text
                        size={fontSize}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'text',
                        })}
                        className={cx('secondLineText')}
                    >
                        {text}
                    </Text>
                </Flex>
            </Button>
        </div>
    );
};

interface IPricePlusButtonProps extends IPriceButtonProps {
    plusPoints?: number;
}

export const PricePlusButton: React.FC<IPricePlusButtonProps> = ({
    plusPoints,
    ...rest
}) => (
    <BorderPositioned
        badge={plusPoints && <PlusColoredLabel count={plusPoints} />}
        direction="top-right-plus"
    >
        <PriceButton {...rest} />
    </BorderPositioned>
);

interface IVariantPriceButtonProps
    extends IPricePlusButtonProps,
        IWithQaAttributes {
    partnerFontSize?: TTextSize;
    buttonRef?: RefObject<HTMLElement>;
    showPartner?: boolean;
}

export const VariantPriceButton: React.FC<IVariantPriceButtonProps> = ({
    price,
    partnerFontSize = 's',
    buttonRef,
    showPartner = false,
    onClick,
    ...rest
}) => {
    const priceConverter = useCurrencyConverter();
    const isPreferredCurrency = priceConverter.isPreferredCurrency(
        price.tariff.currency,
    );

    const partner = useMemo(() => {
        if (showPartner && price.boy) return i18nBlockVariant.boyTitle();

        if (!isPreferredCurrency) {
            return formatPrice({
                value: price.tariff.value,
                currency: price.tariff.currency,
                isCurrencyShown: true,
                isRound: false,
                postfix: showPartner
                    ? i18nBlockPrice.priceOn({
                          partner: price.partner.siteUrl || price.partner.title,
                      })
                    : undefined,
            });
        }

        if (showPartner) {
            return price.partner.siteUrl || price.partner.title;
        }

        return undefined;
    }, [price, isPreferredCurrency, showPartner]);

    return (
        <PricePlusButton
            text={partner}
            fontSize={partnerFontSize}
            plusPoints={price.plusPoints}
            onClick={onClick}
            price={price}
            {...rest}
        />
    );
};

interface IRedirectVariantPriceButtonProps
    extends Omit<IVariantPriceButtonProps, 'price'>,
        IWithClassName {
    variant: IResultAviaVariant;
    buttonClassName?: string;
}

const RedirectVariantPriceButton: React.FC<IRedirectVariantPriceButtonProps> =
    ({variant, ...rest}) => (
        <RedirectLink
            className={cx('redirectPriceButton', rest.className)}
            variant={variant}
        >
            <VariantPriceButton price={variant.price} {...rest} />
        </RedirectLink>
    );

export default RedirectVariantPriceButton;
