import flatten from 'lodash/flatten';
import {useMemo} from 'react';

import {isNotNull} from 'types/utilities';
import {IAviaCarryOn} from 'types/avia/common/IAviaCarryOn';
import {ECarryOnType} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import getWorstCarryOn from 'selectors/avia/utils/denormalization/utilities/getWorstCarryOn';

import {CHAR_MULTIPLICATION_SIGN} from 'utilities/strings/charCodes';

export function getCarryOnFromVariant(
    variant: IResultAviaVariant | null,
): IAviaCarryOn {
    if (!variant) {
        return {
            limitedCarryOn: false,
            places: 0,
            weight: undefined,
            included: false,
        };
    }

    const {carryOn} = variant.price;
    const flatCarryOnInfoByFlights = flatten(
        variant.route.map((flights, i) => {
            return flights.map<Nullable<IAviaCarryOn>>((flight, j) => {
                if (carryOn[i][j]) {
                    return carryOn[i][j];
                }

                const companyTariffHasCarryOn = flight.companyTariff?.carryon;
                const companyTariffCarryOnWeight =
                    flight.companyTariff?.carryonNorm;

                if (companyTariffHasCarryOn) {
                    return {
                        limitedCarryOn: false,
                        places: 1,
                        weight: companyTariffCarryOnWeight || undefined,
                        included: true,
                    };
                }

                const aviaCompany = flight.aviaCompany;

                if (!aviaCompany) {
                    return {
                        limitedCarryOn: false,
                        places: 0,
                        weight: undefined,
                        included: false,
                    };
                }

                const aviaCompanyCarryOnWidth = aviaCompany.carryonWidth;

                const limitedCarryOn =
                    aviaCompany.carryonSizeBucket === ECarryOnType.SMALL ??
                    false;

                if (limitedCarryOn) {
                    const size = [
                        aviaCompany.carryonLength,
                        aviaCompany.carryonWidth,
                        aviaCompany.carryonHeight,
                    ]
                        .filter(Boolean)
                        .join(CHAR_MULTIPLICATION_SIGN);

                    if (size) {
                        return {
                            limitedCarryOn: true,
                            places: aviaCompanyCarryOnWidth ? 1 : 0,
                            weight: undefined,
                            size,
                            included: Boolean(aviaCompanyCarryOnWidth),
                        };
                    }
                }

                return {
                    limitedCarryOn: false,
                    places: aviaCompanyCarryOnWidth ? 1 : 0,
                    weight: undefined,
                    included: Boolean(aviaCompanyCarryOnWidth),
                };
            });
        }),
    );

    if (!flatCarryOnInfoByFlights.length) {
        return {
            limitedCarryOn: false,
            places: 0,
            weight: undefined,
            included: false,
        };
    }

    return getWorstCarryOn(flatCarryOnInfoByFlights.filter(isNotNull));
}

export default function useCarryOnInfo(
    variant: IResultAviaVariant | null,
): IAviaCarryOn {
    return useMemo(() => getCarryOnFromVariant(variant), [variant]);
}
