import {ReactElement, useCallback, useMemo} from 'react';
import {uniq} from 'lodash';

import {ITDPartner} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {TAviaSearchPartnersFilter} from 'reducers/avia/search/results/filters/reducer';

import * as keyset from 'i18n/avia-search';

import Checkbox from 'components/Checkbox/Checkbox';
import Flex from 'components/Flex/Flex';
import {AviaPartnerCheckbox} from 'projects/avia/components/AviaSearchPartnersFilter/AviaPartnerCheckbox';

import cx from './AviaSearchPartnersFilterPopup.scss';

interface IAviaSearchPartnersFilterPopupProps {
    partners: ITDPartner[];
    disabledPartners: Record<string, boolean>;
    value: TAviaSearchPartnersFilter;
    onChange(value: TAviaSearchPartnersFilter): void;
}

export function AviaSearchPartnersFilterPopup({
    disabledPartners,
    onChange,
    partners,
    value,
}: IAviaSearchPartnersFilterPopupProps): ReactElement {
    const handleChange = useCallback(
        (code: string, checked: boolean) => {
            if (checked) {
                onChange(uniq(value.concat(code)));
            } else {
                onChange(value.filter(v => v !== code));
            }
        },
        [value, onChange],
    );

    const allChecked = useMemo(
        () =>
            partners
                .filter(p => !disabledPartners[p.code])
                .every(partner => value.includes(partner.code)),
        [disabledPartners, partners, value],
    );

    const handleAllChange = useCallback(() => {
        if (allChecked) {
            onChange([]);
        } else {
            onChange(
                partners
                    .map(p => p.code)
                    .filter(code => !disabledPartners[code]),
            );
        }
    }, [allChecked, disabledPartners, onChange, partners]);

    return (
        <Flex between="2" flexDirection="column" className={cx('wrapper')}>
            <Checkbox
                size="m"
                label={keyset.filterDotPartnersDotAllPartnersButtonText()}
                checked={allChecked}
                onChange={handleAllChange}
                className={cx('check')}
            />
            {partners.map(partner => (
                <AviaPartnerCheckbox
                    key={partner.code}
                    disabled={disabledPartners[partner.code]}
                    partner={partner}
                    checked={value.includes(partner.code)}
                    onChange={handleChange}
                />
            ))}
        </Flex>
    );
}
