import React, {useCallback, useRef} from 'react';

import {
    IAviaSearchSegmentsAirportsFilter,
    IAviaSearchSegmentsDirectionAirportsFilter,
    TAviaSearchAirportsFilterKeys,
} from 'reducers/avia/search/results/filters/reducer';

import {
    DisabledSegmentStationsViewState,
    AirportsFilterSettlements,
} from 'selectors/avia/search/filters/airportsFilter';

import {SegmentStations} from 'projects/avia/lib/search/stationMapsToStations';
import {useMobile} from 'utilities/hooks/useMobile';

import * as i18nBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import Intersperse from 'components/Intersperse/Intersperse';
import Separator from 'components/Separator/Separator';
import Text from 'components/Text/Text';

import {AirportsFilterDirectionGroup} from './AirportsFilterDirectionGroup';

interface IAirportsFilterPopupProps {
    value: IAviaSearchSegmentsAirportsFilter[];
    onChange: (value: IAviaSearchSegmentsDirectionAirportsFilter[]) => void;
    stationsBySegment: SegmentStations[];
    disabledStations: DisabledSegmentStationsViewState[];
    settlements: AirportsFilterSettlements[];
}

const FILTER_KEYS: TAviaSearchAirportsFilterKeys[] = ['departure', 'arrival'];

export const AirportsFilterPopup = React.memo(function AirportsFilterPopup({
    value,
    onChange,
    stationsBySegment,
    disabledStations,
    settlements,
}: IAirportsFilterPopupProps) {
    const isMobile = useMobile();

    /**
     * Реф нужен для оптимизация handleChange
     */
    const valueRef = useRef(value);

    valueRef.current = value;

    const handleChange = useCallback(
        (
            direction: TAviaSearchAirportsFilterKeys,
            stationId: number,
            segmentIdx: number,
            checked: boolean,
        ) => {
            const currentValue = valueRef.current;
            const segmentFilter = currentValue[segmentIdx] || {
                arrival: [],
                departure: [],
                transfers: [],
            };
            const nextSegmentFilter = {
                ...segmentFilter,
                [direction]: checked
                    ? segmentFilter[direction].concat(stationId)
                    : segmentFilter[direction].filter(id => id !== stationId),
            };
            const nextFilterValue = currentValue.slice();

            nextFilterValue[segmentIdx] = nextSegmentFilter;
            onChange(nextFilterValue);
        },
        [onChange],
    );

    return (
        <Box inline={!isMobile} nowrap={!isMobile} between={isMobile ? 6 : 8}>
            <Intersperse
                separator={isMobile ? <></> : <Separator isVertical />}
            >
                {stationsBySegment.map((stations, idx) => (
                    <Box
                        key={idx}
                        between={isMobile ? 2 : 5}
                        style={{minWidth: isMobile ? 'auto' : 300}}
                    >
                        {isMobile ? (
                            <Box textSize="m" textWeight="bold">
                                {getTitleBySegmentIdx(idx)}
                            </Box>
                        ) : (
                            <Text tag="div" size="m" weight="bold">
                                {getTitleBySegmentIdx(idx)}
                            </Text>
                        )}
                        <Box between="8">
                            {FILTER_KEYS.map(type => (
                                <AirportsFilterDirectionGroup
                                    key={type}
                                    label={getFilterGroupLabelByType(
                                        type,
                                        settlements[idx],
                                    )}
                                    type={type}
                                    stations={stations[type]}
                                    segmentIdx={idx}
                                    onChange={handleChange}
                                    segmentFilter={
                                        value[idx] && value[idx][type]
                                    }
                                    disabledStations={
                                        disabledStations[idx] &&
                                        disabledStations[idx][type]
                                    }
                                />
                            ))}
                        </Box>
                    </Box>
                ))}
            </Intersperse>
        </Box>
    );
});

function getTitleBySegmentIdx(idx: number) {
    switch (idx) {
        case 0:
            return i18nBlock.filterDotAirportsDotForward();
        case 1:
            return i18nBlock.filterDotAirportsDotBackward();
        default:
            return '';
    }
}

function getFilterGroupLabelByType(
    type: TAviaSearchAirportsFilterKeys,
    settlements: AirportsFilterSettlements,
): string {
    switch (type) {
        case 'arrival': {
            const {preposition, phraseTo} = settlements.to;
            const title = phraseTo ?? '';

            if (preposition) {
                return i18nBlock.arrivalToWithPreposition({
                    preposition,
                    title,
                });
            }

            return i18nBlock.filterDotAirportsDotArrivalDashTo({
                title,
            });
        }
        case 'departure':
            return i18nBlock.filterDotAirportsDotDepartureDashFrom({
                title: settlements.from.phraseFrom ?? '',
            });
        default:
            return type;
    }
}
