import React, {useCallback} from 'react';

import {IResultAviaStation} from 'selectors/avia/utils/denormalization/station';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Text from 'components/Text/Text';
import Checkbox from 'components/Checkbox/Checkbox';
import Flex from 'components/Flex/Flex';

import cx from './AirportsStationCheckbox.scss';

interface IAirportsStationCheckboxProps {
    station: IResultAviaStation;
    checked: boolean;
    disabled: boolean;
    onChange(checked: boolean, station: IResultAviaStation): void;
}

export const AirportsStationCheckbox = React.memo(
    function AirportsStationCheckbox({
        checked,
        disabled,
        onChange,
        station,
    }: IAirportsStationCheckboxProps) {
        const handleChange = useCallback(
            (event: React.ChangeEvent<HTMLInputElement>) => {
                onChange(event.target.checked, station);
            },
            [onChange, station],
        );

        const {isMobile} = useDeviceType();

        const settlementTitle = station.settlement && station.settlement.title;
        const stationTitle = station.title || settlementTitle;

        const settlementIsShown =
            settlementTitle && settlementTitle !== stationTitle;

        return (
            <Checkbox
                checked={checked}
                onChange={handleChange}
                disabled={disabled}
                className={cx('root')}
                label={
                    <Flex
                        justifyContent="space-between"
                        alignItems="baseline"
                        textSize="m"
                        between="4"
                        inline
                        nowrap
                    >
                        <Flex justifyContent="flex-start" inline nowrap>
                            {stationTitle && (
                                <Text
                                    tagProps={
                                        isMobile ? {} : {title: stationTitle}
                                    }
                                    className={cx({
                                        airportTitle: !isMobile,
                                        airportTitle_single:
                                            !isMobile && !settlementIsShown,
                                    })}
                                >
                                    {stationTitle}
                                </Text>
                            )}
                            {settlementIsShown && (
                                <Text
                                    tagProps={
                                        isMobile ? {} : {title: settlementTitle}
                                    }
                                    className={cx({airportTitle: !isMobile})}
                                    color="secondary"
                                >
                                    , {settlementTitle}
                                </Text>
                            )}
                        </Flex>
                        <Text color="secondary">
                            {station.code.toUpperCase()}
                        </Text>
                    </Flex>
                }
            />
        );
    },
);
