import React, {useCallback} from 'react';

import {IAlliance} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {TCompaniesByAllienceMap} from 'selectors/avia/search/filters/companyFilter';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-search';

import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import RadioButton from 'components/RadioButton/RadioButton';

import cx from './AviaAllienceSelect.scss';

interface IAllienceSelectProps extends IWithQaAttributes {
    alliences: IAlliance[];
    companiesByAllienceMap: TCompaniesByAllienceMap;
    selectedCompanies: number[];
    availableAlliences: Record<number, boolean>;
    onChange(allienceId: number): void;
    children?: never;
}

export function AllienceSelect({
    alliences,
    companiesByAllienceMap,
    selectedCompanies,
    availableAlliences,
    onChange,
    ...props
}: IAllienceSelectProps) {
    const {isDesktop, isMobile} = useDeviceType();
    const handleChange = useCallback(
        (ev: React.ChangeEvent<HTMLInputElement>) => {
            onChange(Number(ev.target.value));
        },
        [onChange],
    );

    const activeAllience = alliences.find(({id}) => {
        const companies = companiesByAllienceMap[id];

        if (!companies) {
            return false;
        }

        if (Object.keys(companies).length !== selectedCompanies.length) {
            return false;
        }

        return selectedCompanies.every(cId => companies[cId]);
    });

    const content = (
        <RadioButton
            className={cx(isMobile && 'container')}
            onChange={handleChange}
            size="l"
            theme="between"
            value={activeAllience ? activeAllience.id : -1}
        >
            <RadioButton.Radio
                value={-1}
                controlAttrs={prepareQaAttributes({
                    parent: props,
                    current: 'allAlliences',
                })}
            >
                {i18nBlock.filterDotCompanyDotAllAlliancesButtonText()}
            </RadioButton.Radio>
            {alliences.map(allience => (
                <RadioButton.Radio
                    key={allience.id}
                    value={allience.id}
                    disabled={!availableAlliences[allience.id]}
                    controlAttrs={prepareQaAttributes({
                        parent: props,
                        current: 'allience',
                        key: allience.id,
                    })}
                >
                    {allience.title}
                </RadioButton.Radio>
            ))}
        </RadioButton>
    );

    if (isDesktop) {
        return content;
    }

    return <HorizontalScroller offset={4}>{content}</HorizontalScroller>;
}
