import React, {FunctionComponent, useCallback, useRef} from 'react';

import {IResultCompany} from 'selectors/avia/utils/denormalization/company';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useEventListener} from 'utilities/hooks/useEventListener';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-search';

import Checkbox from 'components/Checkbox/Checkbox';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Flex from 'components/Flex/Flex';
import LinkButton from 'components/LinkButton/LinkButton';

import AirlineLogo from '../AirlineLogo/AirlineLogo';

import cx from './AviaCompanyCheckbox.scss';

export interface IAviaCompanyCheckboxProps extends IWithQaAttributes {
    checked: boolean;
    disabled?: boolean;
    // обрабатывает изменения в списке чекбоксов для 1ой компании
    onChangeDiff(checked: boolean, companyId: number): void;
    // устанавливает список выбранных чекбоксов
    onChange: (companyIds: number[]) => void;
    company: IResultCompany;
    price?: IPrice | null;
    children?: never;
}

const AviaCompanyCheckbox: FunctionComponent<IAviaCompanyCheckboxProps> = ({
    checked,
    company,
    price,
    disabled,
    onChangeDiff,
    onChange,
    ...props
}) => {
    const deviceType = useDeviceType();
    const {value: isHovered, toggle: toggleStateOnlyButton} = useBoolean(false);
    const toggleVisibilityOnlyButton = useCallback(() => {
        if (disabled) {
            return;
        }

        toggleStateOnlyButton();
    }, [disabled, toggleStateOnlyButton]);
    const rootRef = useRef<HTMLElement>(null);
    const companyId = company.id;
    const handleChangeDiff = useCallback(
        (ev: React.ChangeEvent<HTMLInputElement>) => {
            onChangeDiff(ev.target.checked, companyId);
        },
        [companyId, onChangeDiff],
    );
    const handleChange = useCallback(
        (event: React.MouseEvent) => {
            event.preventDefault();
            event.stopPropagation();
            onChange([companyId]);
        },
        [onChange, companyId],
    );
    const linkIsShown = !disabled && deviceType.isDesktop && isHovered;

    useEventListener('mouseenter', toggleVisibilityOnlyButton, rootRef);
    useEventListener('mouseleave', toggleVisibilityOnlyButton, rootRef);

    return (
        <Checkbox
            className={cx('root')}
            checked={checked}
            disabled={disabled}
            innerRef={rootRef}
            onChange={handleChangeDiff}
            label={
                <Flex alignItems="baseline" textSize="m" inline nowrap>
                    <AirlineLogo
                        className={cx('logo')}
                        airlineTitle={company.title}
                        logoSrc={company.logoSvg}
                        size="s"
                    />
                    <Text
                        className={cx('title')}
                        size="m"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'title',
                        })}
                    >
                        {company.title}
                    </Text>
                    <Flex className={cx('label')} flexDirection="row-reverse">
                        <LinkButton
                            className={cx(!linkIsShown && 'hidden')}
                            onClick={handleChange}
                        >
                            {i18nBlock.filterDotAirportsDotOnly()}
                        </LinkButton>
                        {price && (
                            <Text
                                className={cx(linkIsShown && 'hidden')}
                                color="secondary"
                            >
                                <Price {...price} />
                            </Text>
                        )}
                    </Flex>
                </Flex>
            }
            {...prepareQaAttributes(props)}
        />
    );
};

export default AviaCompanyCheckbox;
