import {useCallback} from 'react';

import {IAlliance} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {TAviaSearchCompanyFilter} from 'reducers/avia/search/results/filters/reducer';

import {TCompaniesByAllienceMap} from 'selectors/avia/search/filters/companyFilter';
import {IResultCompany} from 'selectors/avia/utils/denormalization/company';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-search';

import Checkbox from 'components/Checkbox/Checkbox';
import Separator from 'components/Separator/Separator';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';

import {AllienceSelect} from './AviaAllienceSelect';
import AviaCompanyCheckbox from './AviaCompanyCheckbox';

import cx from './AviaCompanyFilterPopup.scss';

export interface IAviaCompanyFilterPopupProps extends IWithQaAttributes {
    allAlliences: IAlliance[];
    companiesByAllienceMap: TCompaniesByAllienceMap;
    value: TAviaSearchCompanyFilter;
    allCompanies: IResultCompany[];
    availableCompanies: Record<number, IPrice>;
    availableAlliences: Record<number, boolean>;
    onChange(value: TAviaSearchCompanyFilter): void;
    disableCombinations: boolean;
    children?: never;
}

export function AviaCompanyFilterPopup({
    allAlliences: alliences,
    companiesByAllienceMap,
    value,
    allCompanies,
    availableCompanies,
    availableAlliences,
    onChange,
    disableCombinations,
    ...props
}: IAviaCompanyFilterPopupProps): JSX.Element {
    const deviceType = useDeviceType();
    const {isDesktop, isMobile} = deviceType;

    const handleCompaniesIds = useCallback(
        (companiesIds: number[]) => {
            onChange({
                companiesIds,
                combinationsAreEnabled: value.combinationsAreEnabled,
            });
        },
        [onChange, value.combinationsAreEnabled],
    );

    const handleAllienceChange = useCallback(
        allienceId => {
            const companies = companiesByAllienceMap[allienceId];

            if (companies) {
                handleCompaniesIds(Object.keys(companies).map(Number));
            } else {
                handleCompaniesIds([]);
            }
        },
        [companiesByAllienceMap, handleCompaniesIds],
    );

    const handleCompanyChangeDiff = useCallback(
        (checked: boolean, companyId: number) => {
            if (checked && !value.companiesIds.includes(companyId)) {
                handleCompaniesIds(value.companiesIds.concat(companyId));
            } else {
                handleCompaniesIds(
                    value.companiesIds.filter(id => id !== companyId),
                );
            }
        },
        [handleCompaniesIds, value.companiesIds],
    );

    const handleCombinationChange = useCallback(() => {
        onChange({
            companiesIds: value.companiesIds,
            combinationsAreEnabled: !value.combinationsAreEnabled,
        });
    }, [onChange, value]);

    const availableCompaniesCount = Object.keys(availableCompanies).length;
    const allCompaniesIsChecked =
        availableCompaniesCount > 0 &&
        value.companiesIds.length === availableCompaniesCount;

    const handleAllCompaniesChange = useCallback(() => {
        if (allCompaniesIsChecked) {
            handleCompaniesIds([]);
        } else {
            handleCompaniesIds(Object.keys(availableCompanies).map(Number));
        }
    }, [allCompaniesIsChecked, availableCompanies, handleCompaniesIds]);

    return (
        <div className={cx('root', deviceMods('root', deviceType))}>
            {alliences.length > 0 && (
                <AllienceSelect
                    alliences={alliences}
                    availableAlliences={availableAlliences}
                    companiesByAllienceMap={companiesByAllienceMap}
                    selectedCompanies={value.companiesIds}
                    onChange={handleAllienceChange}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'allienceSelect',
                    })}
                />
            )}
            <Checkbox
                className={cx('combination')}
                checked={value.combinationsAreEnabled}
                onChange={handleCombinationChange}
                disabled={disableCombinations}
                label={
                    <Flex flexDirection="column">
                        <Text>
                            {i18nBlock.filterDotCompanyDotCombinationsTitle()}
                        </Text>
                        <Text
                            className={cx('combination_description')}
                            size={isMobile ? 'm' : 's'}
                            color="secondary"
                        >
                            {i18nBlock.filterDotCompanyDotCombinationsDescription(
                                {isDesktop},
                            )}
                        </Text>
                    </Flex>
                }
                {...prepareQaAttributes({
                    parent: props,
                    current: 'combinationAviaCompanies',
                })}
            />
            <Separator className={cx('separator')} />
            <Checkbox
                checked={allCompaniesIsChecked}
                onChange={handleAllCompaniesChange}
                disabled={availableCompaniesCount === 0}
                label={i18nBlock.filterDotCompanyDotAllCompaniesButtonText()}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'selectAllCompanies',
                })}
            />
            <Flex
                above={isMobile ? 8 : 6}
                between={isMobile ? 6 : 3}
                flexDirection="column"
            >
                {allCompanies.map(company => {
                    const price = availableCompanies[company.id];
                    const isDisabled = !price;
                    const isChecked = value.companiesIds.includes(company.id);

                    return (
                        <AviaCompanyCheckbox
                            key={company.id}
                            checked={isChecked}
                            disabled={isDisabled}
                            company={company}
                            onChangeDiff={handleCompanyChangeDiff}
                            onChange={handleCompaniesIds}
                            price={price}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'selectCompany',
                                key: company.id,
                            })}
                        />
                    );
                })}
            </Flex>
        </div>
    );
}
