import React, {RefObject, useCallback} from 'react';
import {useSelector} from 'react-redux';

import {EAviaResultsSortType} from 'reducers/avia/search/results/EAviaResultsSortType';

import {getAviaContext} from 'selectors/avia/aviaSelectors';

import useDelayRender from 'utilities/hooks/useDelayRender';
import {IUseAviaTotalFilters} from 'projects/avia/lib/hooks/filters/total';
import {useExperiments} from 'utilities/hooks/useExperiments';

import * as i18nBlock from 'i18n/avia-search';

import Intersperse from 'components/Intersperse/Intersperse';
import Separator from 'components/Separator/Separator';
import TimeFilter from './components/TimeFilter';
import useResultsCount from 'projects/avia/components/AviaSearchResultsFilters/hooks/useResultsCount';
import {AviaSearchResultsNoTransferFilterFilterNoConnected} from 'projects/avia/components/AviaSearchResultsNoTransferFilter/AviaSearchResultsNoTransferFilterNoConnected';
import {AviaSearchResultsBaggageFilterNoConnected} from 'projects/avia/components/AviaSearchResultsBaggageFilter/AviaSearchResultsBaggageFilterNoConnected';
import {AviaTransferFilterPopupNoConnected} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopupNoConnected';
import {AviaCompanyFilterPopupNoConnected} from 'projects/avia/components/AviaSearchResultsCompanyFilter/AviaCompanyFilterPopupNoConnected';
import {AirportsFilterPopupNoConnected} from 'projects/avia/components/AviaSearchResultsAirportsFilter/AirportsFilterPopupNoConnected';
import {AviaSearchTimeFilterNoConnected} from 'projects/avia/components/AviaSearchResultsTimeFilter/AviaSearchTimeFilterNoConnected';
import {AviaSearchPartnersFilterPopupNoConnected} from 'projects/avia/components/AviaSearchPartnersFilter/AviaSearchPartnersFilterPopupNoConnected';
import AviaSearchResultsSortingFilterPopupConnected from 'projects/avia/components/AviaSearchResultsSortingFilter/AviaSearchResultsSortingFilterPopupConnected';
import QuickFiltersPadder from 'components/FiltersBottomSheet/components/QuickFiltersPadder/QuickFiltersPadder';
import FilterNestedBottomSheet from 'components/FiltersBottomSheet/components/FilterNestedBottomSheet/FilterNestedBottomSheet';

import cx from './FiltersModalContent.scss';

interface IFiltersModalContentProps {
    isRoundtrip: boolean;
    firstBottomSheetRef: RefObject<HTMLDivElement>;
    totalFilters: IUseAviaTotalFilters;
}

const FiltersModalContent: React.FC<IFiltersModalContentProps> = ({
    isRoundtrip,
    firstBottomSheetRef,
    totalFilters: {
        transfersFilter,
        baggageFilter,
        companyFilter,
        airportsFilter,
        timeFilter,
        partnersFilter,
        sortingFilter,
    },
}) => {
    const canRender = useDelayRender();
    const {oneway} = useSelector(getAviaContext);
    const variantsCount = useResultsCount();
    const {aviaNewTouchFilters} = useExperiments();
    const resetTransfer = useCallback(() => {
        airportsFilter.resetTransferAirports();
        transfersFilter.reset();
    }, [transfersFilter, airportsFilter]);
    const resetCompany = useCallback(
        () => companyFilter.reset(),
        [companyFilter],
    );
    const resetAirports = useCallback(
        () => airportsFilter.resetDirectionAirports(),
        [airportsFilter],
    );
    const resetTime = useCallback(() => timeFilter.reset(), [timeFilter]);
    const resetPartners = useCallback(
        () => partnersFilter.reset(),
        [partnersFilter],
    );

    if (!canRender) {
        return null;
    }

    const submitLabel = variantsCount
        ? i18nBlock.filterDotDone()
        : i18nBlock.filterDashNoDashVariants();

    const sheetCommonProps = {
        submitButtonDisabled: !variantsCount,
        variantsCount,
        submitButtonLabel: submitLabel,
        firstBottomSheetRef,
    };

    return (
        <div className={cx('scrollableContent')}>
            <Intersperse separator={<Separator />}>
                <QuickFiltersPadder padding="none">
                    <AviaSearchResultsNoTransferFilterFilterNoConnected
                        isNewFilters
                        transfersFilter={transfersFilter}
                    />
                    <AviaSearchResultsBaggageFilterNoConnected
                        isNewFilters
                        baggageFilter={baggageFilter}
                    />
                </QuickFiltersPadder>
                {aviaNewTouchFilters && (
                    <FilterNestedBottomSheet
                        {...sheetCommonProps}
                        label={i18nBlock.filterDotSortingTitle()}
                        withReset={false}
                        subLabel={
                            sortingFilter.sorting.type ===
                            EAviaResultsSortType.INTEREST
                                ? undefined
                                : sortingFilter.sortingName?.toLocaleLowerCase()
                        }
                    >
                        <AviaSearchResultsSortingFilterPopupConnected />
                    </FilterNestedBottomSheet>
                )}
                <FilterNestedBottomSheet
                    {...sheetCommonProps}
                    label={i18nBlock.filterDotTransfersDashPlaceholder()}
                    selectedCount={
                        transfersFilter.filter.count +
                        airportsFilter.filter.transfersCount
                    }
                    onReset={resetTransfer}
                    firstBottomSheetRef={firstBottomSheetRef}
                >
                    <AviaTransferFilterPopupNoConnected
                        transfersFilter={transfersFilter}
                        airportsFilter={airportsFilter}
                    />
                </FilterNestedBottomSheet>

                <FilterNestedBottomSheet
                    {...sheetCommonProps}
                    label={i18nBlock.filterDotCompanyDotFilterButtonText({
                        count: 0,
                    })}
                    selectedCount={companyFilter.filter.count}
                    onReset={resetCompany}
                >
                    <AviaCompanyFilterPopupNoConnected
                        companyFilter={companyFilter}
                        transfersFilter={transfersFilter}
                        oneway={oneway}
                    />
                </FilterNestedBottomSheet>

                <FilterNestedBottomSheet
                    {...sheetCommonProps}
                    label={i18nBlock.filterDotAirportsDotTitle()}
                    selectedCount={airportsFilter.filter.directionsCount}
                    onReset={resetAirports}
                    firstBottomSheetRef={firstBottomSheetRef}
                >
                    <AirportsFilterPopupNoConnected
                        airportsFilter={airportsFilter}
                    />
                </FilterNestedBottomSheet>

                <FilterNestedBottomSheet
                    {...sheetCommonProps}
                    label={i18nBlock.filterDotTimeDotNewTitle()}
                    selectedCount={timeFilter.filter.count}
                    onReset={resetTime}
                >
                    <AviaSearchTimeFilterNoConnected
                        isRoundtrip={isRoundtrip}
                        wrapFilter={TimeFilter}
                        timeFilter={timeFilter}
                    />
                </FilterNestedBottomSheet>

                <FilterNestedBottomSheet
                    {...sheetCommonProps}
                    label={i18nBlock.filterDotPartnersDotFilterButtonText()}
                    selectedCount={partnersFilter.filter.count}
                    onReset={resetPartners}
                >
                    <AviaSearchPartnersFilterPopupNoConnected
                        partnersFilter={partnersFilter}
                    />
                </FilterNestedBottomSheet>
            </Intersperse>
        </div>
    );
};

export default React.memo(FiltersModalContent);
