import React, {useCallback, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {IUseAviaTotalFilters} from 'projects/avia/lib/hooks/filters/total';

import * as i18nBlock from 'i18n/avia-search';

import CheckButton from 'components/CheckButton/CheckButton';
import CornerCounter from 'components/CornerCounter/CornerCounter';
import FilterIcon from 'icons/24/Filter';
import {TButtonSize} from 'components/Button/Button';
import FiltersBottomSheet from 'components/FiltersBottomSheet/FiltersBottomSheet';
import useResultsCount from 'projects/avia/components/AviaSearchResultsFilters/hooks/useResultsCount';

import FiltersModalContent from '../FiltersModalContent/FiltersModalContent';

interface IAviaSearchFiltersTriggerButtonProps extends IWithClassName {
    isRoundtrip: boolean;
    size?: TButtonSize;
    totalFilters: IUseAviaTotalFilters;
}

const AviaSearchFiltersTriggerButton: React.FC<IAviaSearchFiltersTriggerButtonProps> =
    ({isRoundtrip, className, size = 'l', totalFilters}) => {
        const {
            value: filtersModalIsVisible,
            setFalse: closeFiltersModal,
            setTrue: openFiltersModal,
        } = useBoolean(false);

        const {
            activeFiltersCount,
            transfersFilter,
            baggageFilter,
            companyFilter,
            airportsFilter,
            timeFilter,
            partnersFilter,
        } = totalFilters;
        const bottomSheetRef = useRef<HTMLDivElement>(null);

        const handleReset = useCallback(() => {
            transfersFilter.reset();
            baggageFilter.reset();
            companyFilter.reset();
            airportsFilter.reset();
            timeFilter.reset();
            partnersFilter.reset();
        }, [
            transfersFilter,
            baggageFilter,
            companyFilter,
            airportsFilter,
            timeFilter,
            partnersFilter,
        ]);

        const variantsCount = useResultsCount();

        return (
            <>
                <CornerCounter count={totalFilters.activeFiltersCount}>
                    <CheckButton
                        className={className}
                        icon={<FilterIcon />}
                        checked={totalFilters.activeFiltersCount > 0}
                        onClick={openFiltersModal}
                        size={size}
                    />
                </CornerCounter>

                <FiltersBottomSheet
                    header={i18nBlock.filters()}
                    resetButtonDisabled={activeFiltersCount === 0}
                    onReset={handleReset}
                    isOpened={filtersModalIsVisible}
                    onClose={closeFiltersModal}
                    outerSheetRef={bottomSheetRef}
                    onSubmit={closeFiltersModal}
                    submitButtonDisabled={!variantsCount}
                    submitLabel={i18nBlock.filtersDotShowVariants({
                        variantsCount,
                    })}
                >
                    <FiltersModalContent
                        isRoundtrip={isRoundtrip}
                        firstBottomSheetRef={bottomSheetRef}
                        totalFilters={totalFilters}
                    />
                </FiltersBottomSheet>
            </>
        );
    };

export default React.memo(AviaSearchFiltersTriggerButton);
