import {ComponentProps, FunctionComponent, useCallback, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {EAviaResultsSortType} from 'reducers/avia/search/results/EAviaResultsSortType';

import {aviaAeroflotPlusSelector} from 'projects/avia/selectors/aviaAeroflotPlusEnabledSelector';
import {hasVariantWithPlusPointsSelector} from 'selectors/avia/search/hasVariantWithPlusPointsSelector';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {ESortDirection} from 'projects/avia/lib/comparators/TComparator';
import {useExperiments} from 'utilities/hooks/useExperiments';

import * as i18nBlock from 'i18n/avia-search';

import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import Select from 'components/Select/Select';
import SortAscendingIcon from 'icons/16/SortAscending';
import SortDescendingIcon from 'icons/16/SortDescending';

import {AviaSearchResultPlusPointsFilter} from '../AviaSearchResultPlusPointsFilter/AviaSearchResultPlusPointsFilter';

import cx from './AviaSearchResultsSortingSelect.scss';

const {ARRIVAL, DEPARTURE, DURATION, PRICE, INTEREST} = EAviaResultsSortType;

const TYPE_GROUP = [INTEREST, PRICE, DEPARTURE, ARRIVAL, DURATION];
const TYPE_GROUP_DEFAULT_PRICE_EXP = [
    PRICE,
    INTEREST,
    DEPARTURE,
    ARRIVAL,
    DURATION,
];

const getItems = (isAviaDefaultSortPrice: boolean) =>
    (isAviaDefaultSortPrice ? TYPE_GROUP_DEFAULT_PRICE_EXP : TYPE_GROUP).map(
        type => ({
            value: type,
            data: getTextBySortType(type as EAviaResultsSortType),
        }),
    );

export type TAviaSearchResultsSortingSelectValue = [
    EAviaResultsSortType,
    ESortDirection,
];

interface IAviaSearchResultsSortingSelectProps extends IWithQaAttributes {
    value: [EAviaResultsSortType, ESortDirection];
    onChange: (values: [EAviaResultsSortType, ESortDirection]) => void;
    size?: ComponentProps<typeof Select>['size'];
}

const AviaSearchResultsSortingSelect: FunctionComponent<IAviaSearchResultsSortingSelectProps> =
    props => {
        const aviaAeroflotPlus = useSelector(aviaAeroflotPlusSelector);
        const hasVariantWithPlusPoints = useSelector(
            hasVariantWithPlusPointsSelector,
        );
        const deviceType = useDeviceType();
        const {
            value: [sortType, sortDirection],
            onChange,
            size,
        } = props;

        const rootQa = getQa(props);

        if (!sortType || !sortDirection) {
            throw new TypeError(
                'Value must be type of [EAviaResultsSortType, ESortDirection]',
            );
        }

        const {aviaDefaultSortPrice} = useExperiments();

        const items = useMemo(
            () => getItems(Boolean(aviaDefaultSortPrice)),
            [aviaDefaultSortPrice],
        );

        const handleSortChange = useCallback(() => {
            onChange([
                sortType,
                sortDirection === ESortDirection.ASCENDING
                    ? ESortDirection.DESCENDING
                    : ESortDirection.ASCENDING,
            ]);
        }, [onChange, sortType, sortDirection]);

        const handleSortTypeChange = useCallback(
            (type: EAviaResultsSortType) => {
                onChange([type, sortDirection]);
            },
            [onChange, sortDirection],
        );

        const sortIcon = useMemo(
            () =>
                sortDirection === ESortDirection.ASCENDING ? (
                    <SortAscendingIcon
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'ascIcon',
                        })}
                    />
                ) : (
                    <SortDescendingIcon
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'descIcon',
                        })}
                    />
                ),
            [sortDirection, rootQa],
        );

        return (
            <Flex
                className={cx(deviceMods('root', deviceType))}
                between="1"
                inline
                nowrap
            >
                <Button
                    icon={sortIcon}
                    onClick={handleSortChange}
                    size={size}
                    {...prepareQaAttributes({parent: rootQa, current: 'order'})}
                />
                <div className={cx('selectWrapper')}>
                    <Select
                        options={items}
                        value={sortType}
                        onChange={handleSortTypeChange}
                        width="max"
                        size={size}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'type',
                        })}
                    />
                </div>
                {aviaAeroflotPlus &&
                    hasVariantWithPlusPoints &&
                    deviceType.isDesktop && (
                        <AviaSearchResultPlusPointsFilter size={size} />
                    )}
            </Flex>
        );
    };

function getTextBySortType(sortType: EAviaResultsSortType): string {
    switch (sortType) {
        case PRICE:
            return i18nBlock.sortDotByDashPrice();
        case DEPARTURE:
            return i18nBlock.sortDotByDashDeparture();
        case ARRIVAL:
            return i18nBlock.sortDotByDashArrival();
        case DURATION:
            return i18nBlock.sortDotByDashDuration();
        case INTEREST:
            return i18nBlock.sortDotByDashInterest();
    }
}

export default AviaSearchResultsSortingSelect;
