import React, {useCallback} from 'react';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useAviaTimeFilter} from 'projects/avia/lib/hooks/filters/time';

import * as i18nBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import Separator from 'components/Separator/Separator';
import DropdownFilter from 'components/DropdownFilter/DropdownFilter';
import FilterHeader from 'components/FiltersBottomSheet/components/FilterHeader/FilterHeader';
import Text from 'components/Text/Text';

import {AviaSearchTimeFilterNoConnected} from './AviaSearchTimeFilterNoConnected';

function renderFilter(label: string, filter: React.ReactNode): React.ReactNode {
    return (
        <Box between={5}>
            <Text tag="h2" size="m" weight="bold">
                {label}
            </Text>
            {filter}
        </Box>
    );
}

interface IAviaSearchResultsTimeFilterProps extends IWithQaAttributes {
    isActive: boolean;
    isRoundtrip: boolean;
    showHeader?: boolean;
    onClickButton?: () => void; // Клик по кнопке-открывашке попапа
    onChangeFilter?: () => void; // Изменение фильтра
    onChange?: () => void; // Любое изменение фильтра, в т.ч и сброс
}

const AviaSearchResultsTimeFilter: React.FC<IAviaSearchResultsTimeFilterProps> =
    ({
        isActive,
        isRoundtrip,
        onClickButton,
        onChangeFilter,
        showHeader,
        onChange,
        ...rest
    }) => {
        const timeFilter = useAviaTimeFilter();

        const handleOnReset = useCallback(() => {
            onChange?.();

            timeFilter.reset();
        }, [onChange, timeFilter]);

        const handleOnChangeFilter = useCallback(() => {
            onChange?.();

            onChangeFilter?.();
        }, [onChangeFilter, onChange]);

        return (
            <DropdownFilter
                label={i18nBlock.filterDotTimeSimpleDotTitle()}
                checked={isActive}
                onClickButton={onClickButton}
                {...prepareQaAttributes(rest)}
            >
                {showHeader && (
                    <FilterHeader
                        label={i18nBlock.filterDotTimeDotNewTitle()}
                        buttonResetDisabled={!timeFilter.filter.isActive}
                        onReset={handleOnReset}
                    />
                )}
                <Box inline nowrap between={8}>
                    <AviaSearchTimeFilterNoConnected
                        timeFilter={timeFilter}
                        wrapFilter={renderFilter}
                        isRoundtrip={isRoundtrip}
                        separator={<Separator isVertical />}
                        onChange={handleOnChangeFilter}
                    />
                </Box>
            </DropdownFilter>
        );
    };

export default React.memo(AviaSearchResultsTimeFilter);
