import {useCallback, useRef} from 'react';

import {
    EAviaSearchTimeFilter,
    IAviaSearchTimeFilter,
} from 'reducers/avia/search/results/filters/reducer';

import {enable, disable, isEnabled, Flag} from 'utilities/flags/flags';
import {useMobile} from 'utilities/hooks/useMobile';

import * as i18nBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';

import {TimeFilterCheckbox} from './TimeFilterCheckbox';

const DATE_TIMES = [
    EAviaSearchTimeFilter.NIGHT,
    EAviaSearchTimeFilter.MORNING,
    EAviaSearchTimeFilter.DAY,
    EAviaSearchTimeFilter.EVENING,
];

interface IDisabledTimeFilters {
    arrival: Flag<EAviaSearchTimeFilter>;
    departure: Flag<EAviaSearchTimeFilter>;
}

interface IAviaTimeFilterPopupProps {
    value: IAviaSearchTimeFilter;
    segmentIdx: number;
    onChange(value: IAviaSearchTimeFilter, segmentIdx: number): void;
    disabled: IDisabledTimeFilters;
}

export function AviaTimeFilterPopup({
    value,
    onChange,
    disabled,
    segmentIdx,
}: IAviaTimeFilterPopupProps): JSX.Element {
    const valueRef = useRef(value);

    valueRef.current = value;

    const isMobile = useMobile();

    const handleDepartureChange = useCallback(
        (checked: boolean, dateTime: EAviaSearchTimeFilter) => {
            onChange(
                {
                    arrival: valueRef.current.arrival,
                    departure: checked
                        ? enable(valueRef.current.departure, dateTime)
                        : disable(valueRef.current.departure, dateTime),
                },
                segmentIdx,
            );
        },
        [onChange, segmentIdx],
    );

    const handleArrivalChange = useCallback(
        (checked: boolean, dateTime: EAviaSearchTimeFilter) => {
            onChange(
                {
                    departure: valueRef.current.departure,
                    arrival: checked
                        ? enable(valueRef.current.arrival, dateTime)
                        : disable(valueRef.current.arrival, dateTime),
                },
                segmentIdx,
            );
        },
        [onChange, segmentIdx],
    );

    return (
        <Box between="6" style={{width: isMobile ? 'auto' : 236}}>
            <div>
                <Box
                    textSize="m"
                    textWeight="normal"
                    below="3"
                    textColor={isMobile ? 'secondary' : 'primary'}
                >
                    {i18nBlock.filterDotTimeDashDeparture()}
                </Box>
                <Flex between="2" flexDirection="column">
                    {DATE_TIMES.map(dateTime => {
                        const isChecked = isEnabled(value.departure, dateTime);
                        const isDisabled =
                            !isChecked &&
                            isEnabled(disabled.departure, dateTime);

                        return (
                            <TimeFilterCheckbox
                                key={dateTime}
                                dateTime={dateTime}
                                checked={isChecked}
                                disabled={isDisabled}
                                onChange={handleDepartureChange}
                            />
                        );
                    })}
                </Flex>
            </div>
            <div>
                <Box
                    textSize="m"
                    textWeight="normal"
                    below="3"
                    textColor={isMobile ? 'secondary' : 'primary'}
                >
                    {i18nBlock.filterDotTimeDashArrival()}
                </Box>
                <Flex between="2" flexDirection="column">
                    {DATE_TIMES.map(dateTime => {
                        const isChecked = isEnabled(value.arrival, dateTime);
                        const isDisabled =
                            !isChecked && isEnabled(disabled.arrival, dateTime);

                        return (
                            <TimeFilterCheckbox
                                key={dateTime}
                                dateTime={dateTime}
                                checked={isChecked}
                                disabled={isDisabled}
                                onChange={handleArrivalChange}
                            />
                        );
                    })}
                </Flex>
            </div>
        </Box>
    );
}
