import {useCallback, useMemo} from 'react';

import {EAviaSearchTimeFilter} from 'reducers/avia/search/results/filters/reducer';

import * as keyset from 'i18n/avia-search';

import Checkbox from 'components/Checkbox/Checkbox';
import Box from 'components/Box/Box';

import cx from './TimeFilterCheckbox.scss';

export interface ITimeFilterCheckboxProps {
    dateTime: EAviaSearchTimeFilter;
    checked: boolean;
    disabled?: boolean;
    onChange(checked: boolean, dateTime: EAviaSearchTimeFilter): void;
}

export function TimeFilterCheckbox({
    checked,
    dateTime,
    disabled,
    onChange,
}: ITimeFilterCheckboxProps) {
    const handleChange = useCallback(
        ev => onChange(ev.target.checked, dateTime),
        [dateTime, onChange],
    );

    const [label, period] = useMemo(() => {
        return getDayTimeLabelAndPeriod(dateTime);
    }, [dateTime]);

    return (
        <Checkbox
            className={cx('root')}
            checked={checked}
            disabled={disabled}
            onChange={handleChange}
            label={
                <div className={cx('label')}>
                    <Box textSize="m">{label}</Box>
                    <Box textColor="secondary" textSize="s">
                        {period}
                    </Box>
                </div>
            }
        />
    );
}

export function getDayTimeLabelAndPeriod(
    dateTime: EAviaSearchTimeFilter,
): [string, string] {
    if (dateTime === EAviaSearchTimeFilter.MORNING) {
        return [
            keyset.filterDotTimeDotTimeDashOfDashDayDashMorning(),
            '06:00—12:00',
        ];
    }

    if (dateTime === EAviaSearchTimeFilter.DAY) {
        return [
            keyset.filterDotTimeDotTimeDashOfDashDayDashDay(),
            '12:00—18:00',
        ];
    }

    if (dateTime === EAviaSearchTimeFilter.EVENING) {
        return [
            keyset.filterDotTimeDotTimeDashOfDashDayDashEvening(),
            '18:00—00:00',
        ];
    }

    return [keyset.filterDotTimeDotTimeDashOfDashDayDashNight(), '00:00—06:00'];
}
