import React, {useRef} from 'react';

import {
    AVIA_TRANSFER_FILTER_NO_TRANSFERS,
    AVIA_TRANSFER_FILTERS_REST_ITEMS,
} from 'projects/avia/components/AviaSearchResultsTransferFilter/constants';

import {EAviaSearchTransferFilter} from 'reducers/avia/search/results/filters/reducer';

import {Flag, isEnabled, enable, disable} from 'utilities/flags/flags';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModDesktop} from 'utilities/stylesUtils';
import {IUseAviaAirportsFilter} from 'projects/avia/lib/hooks/filters/airports';

import Separator from 'components/Separator/Separator';
import {AviaRange} from './components/AviaRange/AviaRange';
import {TransferFilterCheckbox} from 'projects/avia/components/AviaSearchResultsTransferFilter/TransferFilterCheckbox';
import {AviaCities} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopup/components/AviaCities/AviaCities';
import Flex from 'components/Flex/Flex';

import cx from './AviaTransferFilterPopup.scss';

interface IAviaTransferFilterPopupProps {
    hideRange?: boolean;
    hideTransferAirports?: boolean;
    hasTransfers?: boolean;
    transfersValue: Flag<EAviaSearchTransferFilter>;
    transfersDisabled?: Flag<EAviaSearchTransferFilter>;
    onChangeTransfers(value: Flag<EAviaSearchTransferFilter>): void;
    range?: [number, number] | null;
    minDuration?: number;
    maxDuration?: number;
    onRangeChange?: (range: [number, number]) => void;
    airportsFilter?: IUseAviaAirportsFilter;
    children?: never;
    onAirportsChange?: () => void;
}

export function AviaTransferFilterPopup({
    hideRange,
    hideTransferAirports,
    hasTransfers,
    transfersValue,
    transfersDisabled = 0 as Flag<EAviaSearchTransferFilter>,
    onChangeTransfers,
    range,
    minDuration = 0,
    maxDuration = 0,
    onRangeChange,
    airportsFilter,
    onAirportsChange,
}: IAviaTransferFilterPopupProps): JSX.Element {
    const transfersIsDisabled = isEnabled(
        transfersValue,
        EAviaSearchTransferFilter.NO_TRANSFERS,
    );
    const valueRef = useRef(transfersValue);

    valueRef.current = transfersValue;

    const handleChange = React.useCallback(
        (checked: boolean, val: EAviaSearchTransferFilter) => {
            const nextValue = checked
                ? enable(valueRef.current, val)
                : disable(valueRef.current, val);

            onChangeTransfers(nextValue);
        },
        [onChangeTransfers],
    );

    const deviceType = useDeviceType();
    const {isMobile} = deviceType;

    return (
        <div className={cx(deviceModDesktop('root', deviceType))}>
            <TransferFilterCheckbox
                value={AVIA_TRANSFER_FILTER_NO_TRANSFERS.value}
                label={AVIA_TRANSFER_FILTER_NO_TRANSFERS.label}
                checked={isEnabled(
                    transfersValue,
                    AVIA_TRANSFER_FILTER_NO_TRANSFERS.value,
                )}
                key={AVIA_TRANSFER_FILTER_NO_TRANSFERS.value}
                onChange={handleChange}
                disabled={isEnabled(
                    transfersDisabled,
                    AVIA_TRANSFER_FILTER_NO_TRANSFERS.value,
                )}
            />
            <Separator margin={isMobile ? 6 : 5} />
            <Flex between={isMobile ? 6 : 3} flexDirection="column">
                {AVIA_TRANSFER_FILTERS_REST_ITEMS.map(item => (
                    <TransferFilterCheckbox
                        value={item.value}
                        label={item.label}
                        checked={isEnabled(transfersValue, item.value)}
                        key={item.value}
                        onChange={handleChange}
                        disabled={
                            transfersIsDisabled ||
                            isEnabled(transfersDisabled, item.value)
                        }
                    />
                ))}
            </Flex>
            {!hideRange && hasTransfers && (
                <AviaRange
                    range={range}
                    value={transfersValue}
                    minDuration={minDuration}
                    maxDuration={maxDuration}
                    onRangeChange={onRangeChange}
                />
            )}
            {!hideTransferAirports && airportsFilter && (
                <AviaCities
                    airportsFilter={airportsFilter}
                    transfersIsDisabled={transfersIsDisabled}
                    onChange={onAirportsChange}
                />
            )}
        </div>
    );
}
