import {useCallback, useMemo, useRef, useState} from 'react';
import {useSelector} from 'react-redux';

import {EOneWay} from 'server/services/AviaSearchService/types/IAviaParams';

import {getAviaContext} from 'selectors/avia/aviaSelectors';

import {stationMapsToStations} from 'projects/avia/lib/search/stationMapsToStations';
import {IUseAviaAirportsFilter} from 'projects/avia/lib/hooks/filters/airports';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {preparedCities} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopup/components/AviaCityDirectionGroup/helpers/preparedCities';

import * as i18nBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import Tabs from 'components/Tabs/Tabs';
import Text from 'components/Text/Text';
import {AviaCityDirectionGroup} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopup/components/AviaCityDirectionGroup/AviaCityDirectionGroup';

import cx from './AviaCities.scss';

interface IAviaCitiesProps {
    airportsFilter: IUseAviaAirportsFilter;
    transfersIsDisabled: boolean;
    onChange?: () => void;
}

const tabs = [
    {
        id: '0',
        name: i18nBlock.filterDotAirportsDotForward(),
    },
    {
        id: '1',
        name: i18nBlock.filterDotAirportsDotBackward(),
    },
];

export function AviaCities({
    airportsFilter: {
        filter: {
            value,
            viewData: {disabledStations, stations},
        },
        handleTransferAirports: onFilterChange,
    },
    transfersIsDisabled,
    onChange,
}: IAviaCitiesProps): JSX.Element | null {
    const deviceType = useDeviceType();
    const {oneway} = useSelector(getAviaContext);
    const [activeTabId, setActiveTabId] = useState(tabs[0].id);
    const stationsBySegment = useMemo(
        () => stations.map(stationMapsToStations),
        [stations],
    );

    /** Реф нужен для оптимизация handleChange */
    const valueRef = useRef(value);

    valueRef.current = value;

    const handleChangeDiff = useCallback(
        (stationIds: number[], segmentIdx: number, checked: boolean) => {
            onChange?.();

            const currentValue = valueRef.current;
            const segmentFilter = currentValue[segmentIdx] || {
                arrival: [],
                departure: [],
                transfers: [],
            };
            const nextSegmentFilter = {
                ...segmentFilter,
                transfers: checked
                    ? segmentFilter.transfers.concat(stationIds)
                    : segmentFilter.transfers.filter(
                          id => !stationIds.includes(id),
                      ),
            };
            const nextFilterValue = currentValue.slice();

            nextFilterValue[segmentIdx] = nextSegmentFilter;
            onFilterChange(nextFilterValue);
        },
        [onChange, onFilterChange],
    );
    const handleChange = useCallback(
        (stationIds: number[], segmentIdx: number) => {
            onChange?.();

            const currentValue = valueRef.current;
            const segmentFilter = currentValue[segmentIdx] || {
                arrival: [],
                departure: [],
                transfers: [],
            };
            const nextSegmentFilter = {
                ...segmentFilter,
                transfers: stationIds,
            };
            const nextFilterValue = currentValue.slice();

            nextFilterValue[segmentIdx] = nextSegmentFilter;
            onFilterChange(nextFilterValue);
        },
        [onChange, onFilterChange],
    );
    const directionToCities = useMemo(
        () =>
            stationsBySegment.map(directionStations =>
                preparedCities(directionStations.transfers),
            ),
        [stationsBySegment],
    );
    const filterIsEmpty = useMemo(() => {
        return (
            directionToCities.reduce(
                (citiesCount, directionStations) =>
                    citiesCount + directionStations.length,
                0,
            ) === 0
        );
    }, [directionToCities]);

    const renderTransfers = useCallback(
        (segmentIdx: number) => {
            return (
                <AviaCityDirectionGroup
                    className={cx('directionGroup')}
                    cities={directionToCities[segmentIdx]}
                    stations={stationsBySegment[segmentIdx].transfers}
                    segmentIdx={segmentIdx}
                    onChangeDiff={handleChangeDiff}
                    onChange={handleChange}
                    segmentFilter={
                        value[segmentIdx] && value[segmentIdx].transfers
                    }
                    disabledStations={
                        disabledStations[segmentIdx] &&
                        disabledStations[segmentIdx].transfers
                    }
                    transfersIsDisabled={transfersIsDisabled}
                />
            );
        },
        [
            stationsBySegment,
            handleChangeDiff,
            disabledStations,
            value,
            handleChange,
            transfersIsDisabled,
            directionToCities,
        ],
    );
    const transfersForward = useMemo(
        () => renderTransfers(0),
        [renderTransfers],
    );
    const transfersBackward = useMemo(() => {
        if (oneway === EOneWay.ONE_WAY) {
            return null;
        }

        return renderTransfers(1);
    }, [renderTransfers, oneway]);

    const onTabChange = useCallback(tab => setActiveTabId(tab.id), []);
    const getTabId = useCallback(tab => tab.id, []);
    const getTabIsDisabled = useCallback(
        () => transfersIsDisabled,
        [transfersIsDisabled],
    );
    const renderTab = useCallback(
        ({tab, tabProps}) => (
            <Tabs.Tab key={tab.id} {...tabProps}>
                {tab.name}
            </Tabs.Tab>
        ),
        [],
    );

    if (filterIsEmpty) {
        return null;
    }

    return (
        <Box className={cx('root', deviceMods('root', deviceType))} above={8}>
            <Text weight="bold" disabled={transfersIsDisabled}>
                {i18nBlock.filterDotTransfersDotAirport()}
            </Text>
            {oneway === EOneWay.ROUND_TRIP && (
                <Tabs
                    border
                    tabs={tabs}
                    activeTabId={activeTabId}
                    onChange={onTabChange}
                    getTabId={getTabId}
                    getTabIsDisabled={getTabIsDisabled}
                    renderTab={renderTab}
                />
            )}
            {activeTabId === '0' ? transfersForward : transfersBackward}
        </Box>
    );
}
