import React, {useCallback, useRef} from 'react';

import {IResultAviaStation} from 'selectors/avia/utils/denormalization/station';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useEventListener} from 'utilities/hooks/useEventListener';

import * as i18nBlock from 'i18n/avia-search';

import Text from 'components/Text/Text';
import Checkbox from 'components/Checkbox/Checkbox';
import Flex from 'components/Flex/Flex';
import LinkButton from 'components/LinkButton/LinkButton';

import cx from './AviaCityCheckbox.scss';

export interface ICityDirectionGroups {
    title: string;
    code: string;
    stations: IResultAviaStation[];
}

interface ICityCheckboxProps {
    city: ICityDirectionGroups;
    checked: boolean;
    disabled: boolean;
    onChangeDiff: (checked: boolean, city: ICityDirectionGroups) => void;
    onChange: (newStations: IResultAviaStation[]) => void;
    transfersIsDisabled: boolean;
}

export const AviaCityCheckbox = React.memo(function AirportsStationCheckbox({
    checked,
    disabled,
    onChangeDiff,
    onChange,
    city,
    transfersIsDisabled,
}: ICityCheckboxProps) {
    const {value: isHovered, toggle: toggleStateOnlyButton} = useBoolean(false);
    const toggleVisibilityOnlyButton = useCallback(() => {
        if (transfersIsDisabled) {
            return;
        }

        toggleStateOnlyButton();
    }, [transfersIsDisabled, toggleStateOnlyButton]);
    const rootRef = useRef<HTMLElement>(null);
    const deviceType = useDeviceType();
    const handleChangeDiff = useCallback(
        (event: React.ChangeEvent<HTMLInputElement>) => {
            onChangeDiff(event.target.checked, city);
        },
        [onChangeDiff, city],
    );
    const handleChange = useCallback(
        (event: React.MouseEvent) => {
            event.preventDefault();
            event.stopPropagation();
            onChange(city.stations);
        },
        [onChange, city],
    );
    const linkIsShown = !disabled && deviceType.isDesktop && isHovered;

    useEventListener('mouseenter', toggleVisibilityOnlyButton, rootRef);
    useEventListener('mouseleave', toggleVisibilityOnlyButton, rootRef);

    return (
        <Checkbox
            className={cx('root')}
            checked={checked}
            disabled={disabled}
            innerRef={rootRef}
            onChange={handleChangeDiff}
            label={
                <Flex
                    justifyContent="space-between"
                    alignItems="baseline"
                    textSize="m"
                    between="4"
                    inline
                    nowrap
                >
                    <div className={cx('title')} title={city.title}>
                        {city.title}
                    </div>
                    <Flex className={cx('label')} flexDirection="row-reverse">
                        <LinkButton
                            className={cx(!linkIsShown && 'hidden')}
                            onClick={handleChange}
                        >
                            {i18nBlock.filterDotAirportsDotOnly()}
                        </LinkButton>
                        <Text
                            className={cx(linkIsShown && 'hidden')}
                            color="secondary"
                        >
                            {city.code.toUpperCase()}
                        </Text>
                    </Flex>
                </Flex>
            }
        />
    );
});
